//--------------------------
struct vec3 {
//--------------------------
	double x, y, z;

	__device__ vec3(double x0 = 0, double y0 = 0, double z0 = 0) { x = x0; y = y0; z = z0; }

	__device__ vec3 operator*(double a) const { return vec3(x * a, y * a, z * a); }
	__device__ vec3 operator/(double a) const { return vec3(x / a, y / a, z / a); }
	__device__ vec3 operator+(const vec3& v) const { return vec3(x + v.x, y + v.y, z + v.z); }
	__device__ vec3 operator-(const vec3& v) const { return vec3(x - v.x, y - v.y, z - v.z); }
	__device__ vec3 operator*(const vec3& v) const { return vec3(x * v.x, y * v.y, z * v.z); }
	__device__ vec3 operator-()  const { return vec3(-x, -y, -z); }
    __device__ double& operator[](int i) { return *(&x + i); }
};

__device__ inline double dot(const vec3& v1, const vec3& v2) { return (v1.x * v2.x + v1.y * v2.y + v1.z * v2.z); }

__device__ inline double length(const vec3& v) { return sqrtf(dot(v, v)); }

__device__ inline vec3 normalize(const vec3& v) { return v * (1 / length(v)); }

__device__ inline vec3 cross(const vec3& v1, const vec3& v2) {
	return vec3(v1.y * v2.z - v1.z * v2.y, v1.z * v2.x - v1.x * v2.z, v1.x * v2.y - v1.y * v2.x);
}

__device__ inline vec3 operator*(double a, const vec3& v) { return vec3(v.x * a, v.y * a, v.z * a); }


__device__ inline double signeddistance(const vec3& planeNN, const vec3& planeP, const vec3& Q) {
    vec3 PQ = Q - planeP;
    return dot(PQ, planeNN);
}

__device__ inline vec3 intersection(const vec3& planeNN, const vec3& planeP, const vec3& Q) {
    double t = (dot(planeNN, Q) - dot(planeNN, planeP)) / (dot(planeNN, planeNN));
    return (Q - planeNN*t);
}

__device__ inline bool intriangle(const vec3& Q, const vec3& A, const vec3& B, const vec3& C) {
    vec3 planeNN = cross(B-A, C-A);
    double len = dot(planeNN, planeNN);
    vec3 aN = cross(C-B, Q-B);
    vec3 bN = cross(A-C, Q-C);
    vec3 cN = cross(B-A, Q-A);
    double alpha = dot(planeNN, aN) / len;
    double betha = dot(planeNN, bN) / len;
    double gamma = dot(planeNN, cN) / len;
    if (0.0 < alpha && alpha < 1.0 && 0.0 < betha && betha < 1.0 && 0.0 < gamma && gamma < 1.0)
        return true;
    return false;
}

__device__ inline int stabil_ep(const vec3& S, const vec3& C, const vec3& D) {
    int cnt = 0;
    vec3 A(0, 0, 0), B(1.0, 0, 0.0);
    // ABC oldal:
    vec3 planeNN = normalize(cross(B-A, C-A));
    vec3 P = intersection(planeNN, A, S);
    if (intriangle(P, A, B, C))
        cnt++;
    
    // BCD oldal:
    planeNN = normalize(cross(B-C, C-D));
    P = intersection(planeNN, B, S);
    if (intriangle(P, D, B, C))
        cnt++;

    // CDA oldal:
    planeNN = normalize(cross(C-A, D-A));
    P = intersection(planeNN, A, S);
    if (intriangle(P, A, D, C))
        cnt++;

    // DAB oldal:
    planeNN = normalize(cross(D-A, B-A));
    P = intersection(planeNN, D, S);
    if (intriangle(P, A, D, B))
        cnt++;
    
    return cnt;
}

__device__ inline bool instabil_ell(const vec3& S, const vec3& X, const vec3& A, const vec3& B, const vec3& C) {
    vec3 planeN = normalize(X - S);
    double signDistanceA = signeddistance(planeN, X, A);
    double signDistanceB = signeddistance(planeN, X, B);
    double signDistanceC = signeddistance(planeN, X, C);
    if (signDistanceA <= 0.0 && signDistanceB <= 0.0 && signDistanceC <= 0.0)
        return true;
    if (signDistanceA > 0.0 && signDistanceB > 0.0 && signDistanceC > 0.0)
        return true;
    return false;
}

__device__ inline int instabil_ep(const vec3& S, const vec3& C, const vec3& D) {
    int cnt = 0;
    vec3 A(0, 0, 0), B(1.0, 0, 0.0);
    // D CSUCSNAL
    if (instabil_ell(S, D, A, B, C))
        cnt++;
    // A CSUCSNAL
    if (instabil_ell(S, A, D, B, C))
        cnt++;
    // B CSUCSNAL
    if (instabil_ell(S, B, A, D, C))
        cnt++;
    // C CSUCSNAL
    if (instabil_ell(S, C, A, B, D))
        cnt++;
    return cnt;
}

__device__ inline vec3 lineintersect(const vec3& S, const vec3& A, const vec3& B) {
    return A + (dot(S - A, B - A) * (B - A)) / dot(B - A, B - A); 
}

__device__ inline bool nyereg_ell(const vec3& S, const vec3& X1, const vec3& X2, const vec3& A, const vec3& B) {
    vec3 intersect = lineintersect(S, X1, X2);
    double KAC = dot(X2 - X1, S - X1);
    double KAB = dot(X2 - X1, X2 - X1);
    if (0.0 < KAC && KAC < KAB) {
        vec3 planeN = normalize(intersect - S);
        double signDistanceA = signeddistance(planeN, intersect, A);
        double signDistanceB = signeddistance(planeN, intersect, B);
        if (signDistanceA <= 0.0 && signDistanceB <= 0.0)
            return true;
        if (signDistanceA > 0.0 && signDistanceB > 0.0)
            return true;
    }
    return false;
}

__device__ inline int nyereg_ep(const vec3& S, const vec3& C, const vec3& D) {
    int cnt = 0;
    vec3 A(0, 0, 0), B(1.0, 0.0, 0.0);

    if (nyereg_ell(S, A, B, C, D))
        cnt++;
    if (nyereg_ell(S, B, C, D, A))
        cnt++;
    if (nyereg_ell(S, C, D, A, B))
        cnt++;
    if (nyereg_ell(S, A, D, C, A))
        cnt++;
    if (nyereg_ell(S, A, C, B, D))
        cnt++;
    if (nyereg_ell(S, D, B, C, A))
        cnt++;

    return cnt;
}

__device__ void ABC_oldal(int v, int w, const vec3& C, const vec3& D, char* egysulyi_mtx) {
    int pos = blockDim.x * blockIdx.x + threadIdx.x;

    vec3 AB(1.0/v, 0.0, 0.0);
    vec3 AC = C/v;

    for (double i = 0.0; i < v; i++)
    {
        for (double j = 0.0; j < v; j++)
        {
            vec3 K = i*AB + j * AC;
            vec3 L = (D - K)/w;
            for (double k = 0.0; k < w; k++)
            {
                vec3 Sv = K + L*k;
                int S = stabil_ep(Sv, C, D);
                int U = instabil_ep(Sv, C, D);
                int H = nyereg_ep(Sv, C, D);

                if (S > 0 && U > 0 && S + U - H == 2)
                    egysulyi_mtx[pos*16+(S-1)*4+(U-1)] = 1;
            }
        }
    }
}

__device__ void BCD_oldal(int v, int w, const vec3& C, const vec3& D, char* egysulyi_mtx) {
    int pos = blockDim.x * blockIdx.x + threadIdx.x;

    vec3 A(0.0, 0.0, 0.0), B(1.0, 0.0, 0.0);
    vec3 BC = (C - B) / v;
    vec3 BD = (D - B) / v;

    for (double i = 0.0; i < v; i++)
    {
        for (double j = 0.0; j < v; j++)
        {
            vec3 K = B + i * BC + j * BD;
            vec3 L = (A - K)/w;
            for (double k = 0.0; k < w; k++)
            {
                vec3 Sv = K + L*k;
                int S = stabil_ep(Sv, C, D);
                int U = instabil_ep(Sv, C, D);
                int H = nyereg_ep(Sv, C, D);

                if (S > 0 && U > 0 && S + U - H == 2)
                    egysulyi_mtx[pos*16+(S-1)*4+(U-1)] = 1;
            }
        }
    }
}

__device__ void CDA_oldal(int v, int w, const vec3& C, const vec3& D, char* egysulyi_mtx) {
    int pos = blockDim.x * blockIdx.x + threadIdx.x;

    vec3 A(0.0, 0.0, 0.0), B(1.0, 0.0, 0.0);
    vec3 CA = (A - C) / v;
    vec3 CD = (D - C) / v;

    for (double i = 0.0; i < v; i++)
    {
        for (double j = 0.0; j < v; j++)
        {
            vec3 K = C + i * CA + j * CD;
            vec3 L = (B - K)/w;
            for (double k = 0.0; k < w; k++)
            {
                vec3 Sv = K + L*k;
                int S = stabil_ep(Sv, C, D);
                int U = instabil_ep(Sv, C, D);
                int H = nyereg_ep(Sv, C, D);

                if (S > 0 && U > 0 && S + U - H == 2)
                    egysulyi_mtx[pos*16+(S-1)*4+(U-1)] = 1;
            }
        }
    }
}

__device__ void DAB_oldal(int v, int w, const vec3& C, const vec3& D, char* egysulyi_mtx) {
    int pos = blockDim.x * blockIdx.x + threadIdx.x;

    vec3 A(0.0, 0.0, 0.0), B(1.0, 0.0, 0.0);
    vec3 DA = (A - D) / v;
    vec3 DB = (B - D) / v;

    for (double i = 0.0; i < v; i++)
    {
        for (double j = 0.0; j < v; j++)
        {
            vec3 K = D + i * DA + j * DB;
            vec3 L = (C - K)/w;
            for (double k = 0.0; k < w; k++)
            {
                vec3 Sv = K + L*k;
                int S = stabil_ep(Sv, C, D);
                int U = instabil_ep(Sv, C, D);
                int H = nyereg_ep(Sv, C, D);

                if (S > 0 && U > 0 && S + U - H == 2)
                    egysulyi_mtx[pos*16+(S-1)*4+(U-1)] = 1;
            }
        }
    }
}

__global__ void gpu_egyensulyi(int v, int w, const double* Cx_arr, const  double* Cy_arr, 
               const double* Dx_arr, const double* Dy_arr, const double* Dz_arr, int size_C, int size_D, int lcm, char* egysulyi_mtx) {
    int pos = blockDim.x * blockIdx.x + threadIdx.x;
    if (pos >= size_C*size_D)
        return;
    if (pos == 0)
        printf("%d", size_C*size_D);
    vec3 C(Cx_arr[pos % size_C], Cy_arr[pos % size_C], 0.0);
    vec3 D(Dx_arr[(pos + pos / lcm) % size_D], Dy_arr[(pos + pos / lcm) % size_D], Dz_arr[(pos + pos / lcm) % size_D]);

    ABC_oldal(v, w, C, D, egysulyi_mtx);
    BCD_oldal(v, w, C, D, egysulyi_mtx);
    CDA_oldal(v, w, C, D, egysulyi_mtx);
    DAB_oldal(v, w, C, D, egysulyi_mtx);
}

__global__ void gpu_pontok(int v, int w, double Cx, double Cy, double Dx, double Dy, double Dz, float* points, char* type) {
    int pos = blockDim.x * blockIdx.x + threadIdx.x * w + threadIdx.y;
    vec3 C(Cx, Cy, 0.0);
    vec3 D(Dx, Dy,  Dz);

    vec3 AB(1.0/v, 0.0, 0.0);
    vec3 AC = C/v;

    vec3 K = ( (double) blockIdx.x) * AB + ((double) threadIdx.x) * AC;
    vec3 L = (D - K)/w;
    vec3 Sv = K + L*((double) threadIdx.y);
    int S = stabil_ep(Sv, C, D);
    int U = instabil_ep(Sv, C, D);
    int H = nyereg_ep(Sv, C, D);
    if (S > 0 && U > 0 && S + U - H == 2) {
        points[3*pos] = Sv.x;
        points[3*pos + 1] = Sv.y;
        points[3*pos + 2] = Sv.z;
        type[pos] = (S-1)*4+(U-1);
    }

    //ABC_oldal(v, w, C, D, egysulyi_mtx);
    //BCD_oldal(v, w, C, D, egysulyi_mtx);
    //CDA_oldal(v, w, C, D, egysulyi_mtx);
    //DAB_oldal(v, w, C, D, egysulyi_mtx);
}