#!/usr/bin/python

from itertools import islice
from socket import gethostname
import argparse
import logging
import os
import pika
import psutil
import time
import adal
from azureperformancecounters import AzureVmPerformanceCounters
from azure.common.credentials import ServicePrincipalCredentials
import azure.mgmt.compute

logger = logging.getLogger(__name__)


class Client:

    env_config = {
        "server_address": "GRAPHITE_HOST",
        "server_port": "GRAPHITE_PORT",
        "amqp_user": "GRAPHITE_AMQP_USER",
        "amqp_pass": "GRAPHITE_AMQP_PASSWORD",
        "amqp_queue": "GRAPHITE_AMQP_QUEUE",
        "amqp_vhost": "GRAPHITE_AMQP_VHOST",
        "subscription_id": "AZURE_SUBSCRIPTION_ID",
        "resource_group": "AZURE_RESOURCE_GROUP",
        "client_id": "AZURE_CLIENT_ID",
        "client_oauth_id": "AZURE_CLIENT_OAUTH_ID",
        "client_secret": "AZURE_CLIENT_SECRET",
        "client_url": "AZURE_CLIENT_URL",
        "tenant_id": "AZURE_TENANT_ID",
    }

    def __init__(self):
        """
        Constructor of the client class that is responsible for handling the
        communication between the graphite server and the data source. In
        order to initialize a client you must have the following
        environmental varriables:
        - GRAPHITE_SERVER_ADDRESS:
        - GRAPHITE_SERVER_PORT:
        - GRAPHITE_AMQP_USER:
        - GRAPHITE_AMQP_PASSWORD:
        - GRAPHITE_AMQP_QUEUE:
        - GRAPHITE_AMQP_VHOST:
        - AZURE_SUBSCRIPTION_ID:
        - AZURE_CLIENT_ID:
        - AZURE_CLIENT_SECRET:
        - AZURE_CLIENT_URL:
        - AZURE_TENANT_ID:
        Missing only one of these variables will cause the client not to work.
        """
        self.name = 'circle.%s' % gethostname()
        for var, env_var in self.env_config.items():
            value = os.getenv(env_var, "")
            if value:
                setattr(self, var, value)
            else:
                raise RuntimeError('%s environment variable missing' % env_var)

        self.credentials = ServicePrincipalCredentials(
            client_id=self.client_id,
            secret=self.client_secret,
            tenant=self.tenant_id,
        )

        self.compute_client = azure.mgmt.compute.ComputeManagementClient(
            self.credentials,
            self.subscription_id,
        )

    def refresh_rest_api_token(self):
        authentication_context = adal.AuthenticationContext(
            'https://login.microsoftonline.com/%s' % (self.client_oauth_id),
        )

        access_token = \
            authentication_context.acquire_token_with_client_credentials(
                'https://management.azure.com/',
                self.client_url, self.client_secret,
            )

        self.access_token = access_token["accessToken"]

    def connect(self):
        """
        This method creates the connection to the queue of the graphite
        server using the environmental variables given in the constructor.
        Returns true if the connection was successful.
        """
        try:
            credentials = pika.PlainCredentials(self.amqp_user, self.amqp_pass)
            params = pika.ConnectionParameters(host=self.server_address,
                                               port=int(self.server_port),
                                               virtual_host=self.amqp_vhost,
                                               credentials=credentials)
            self.connection = pika.BlockingConnection(params)
            self.channel = self.connection.channel()
            logger.info('Connection established to %s.', self.server_address)
        except RuntimeError:
            logger.error('Cannot connect to the server. '
                         'Parameters may be wrong.')
            logger.error("An error has occured while connecting to the server")
            raise
        except:  # FIXME
            logger.error('Cannot connect to the server. There is no one '
                         'listening on the other side.')
            raise

    def disconnect(self):
        """
        Break up the connection to the graphite server. If something went
        wrong while disconnecting it simply cut the connection up.
        """
        try:
            self.channel.close()
            self.connection.close()
        except RuntimeError as e:
            logger.error('An error has occured while disconnecting. %s',
                         unicode(e))
            raise

    def send(self, message):
        """
        Send the message given in the parameters given in the message
        parameter. This function expects that the graphite server want the
        metric name given in the message body. (This option must be enabled
        on the server. Otherwise it can't parse the data sent.)
        """
        body = "\n".join(message)
        try:
            self.channel.basic_publish(exchange=self.amqp_queue,
                                       routing_key='', body=body)
        except:
            logger.error('An error has occured while sending metrics (%dB).',
                         len(body))
            raise

    def _collect_running_vms_for_group(self, group_name):
        """
        Collect running vms for resource group
        :group_name: resource group name to collect running vms for
        """
        logger.info("getting vm list for resurce group '%s'" % (group_name))

        virtual_machine_names = self.compute_client.virtual_machines.list(
            group_name)

        running_virtual_machines = []
        for vm_name in virtual_machine_names:
            logger.info("get state of vm '%s'" % (vm_name.name))
            # get vm by resource group and name
            vm = self.compute_client.virtual_machines.get(
                group_name, vm_name.name, expand="instanceView"
            )

            try:
                # get vm power state
                vm_power_state = (
                    status for status in vm.instance_view.statuses
                    if status.code.startswith("PowerState")
                ).next().code

                if(vm_power_state == "PowerState/running"):
                    running_virtual_machines.append(vm)
            except:
                pass

        return running_virtual_machines

    def datetime_to_time(self, dt):
        """
        Convert datetime to time
        :dt: datetime object to convert to time
        """
        return time.mktime(dt.timetuple()) + dt.microsecond / 1E6

    def collect_vms(self):
        """
        This method is used for fetching vm's information running in
        Azure and using the cmdline parameters calculates different types of
        resource usages about the vms.
        """

        metrics = []
        running_vms = self._collect_running_vms_for_group(self.resource_group)

        for vm in running_vms:
            vm_counters = AzureVmPerformanceCounters(
                self.credentials, self.access_token,
                self.subscription_id, self.resource_group, vm
            ).performance_counters

            if vm_counters.has_key("memory_usage"):
                metrics.append(
                    'vm.%(name)s.memory.usage %(value)f %(time)d' % {
                        'name': vm.name,
                        'value': vm_counters["memory_usage"]["value"],
                        'time': time.time(),
                    }
                )

            if vm_counters.has_key("processor_usage"):
                metrics.append(
                    'vm.%(name)s.cpu.percent %(value)f %(time)d' % {
                        'name': vm.name,
                        'value': vm_counters["processor_usage"]["value"],
                        'time': time.time(),
                    }
                )

            for unit in ["bytes", "packets"]:
                for direction in ["recv", "sent"]:
                    pc_name = "%s_%s" % (unit, direction)
                    if vm_counters.has_key(pc_name):
                        metrics.append(
                            'vm.%(name)s.network.%(metric)s-'
                            '%(interface)s %(data)f %(time)d' % {
                                'name': vm.name,
                                'interface': "eth0",
                                'metric': pc_name,
                                'time': time.time(),
                                'data': vm_counters[pc_name]["value"],
                            }
                        )

        metrics.append(
            '%(host)s.vmcount %(data)d %(time)d' % {
                'host': "azure",
                'data': len(running_vms),
                'time': time.time(),
            }
        )

        logger.info("current metrics:")
        logger.info(metrics)

        return metrics

    @staticmethod
    def _chunker(seq, size):
        """
        Yield seq in size-long chunks.
        """
        for pos in xrange(0, len(seq), size):
            yield islice(seq, pos, pos + size)

    def run(self):
        """
        Call this method to start reporting to the server, it needs the
        metricCollectors parameter that should be provided by the collectables
        modul to work properly.
        """
        self.connect()
        self.processes = {}
        try:
            while True:
                self.refresh_rest_api_token()
                metrics = self.collect_vms()
                if metrics:
                    for chunk in self._chunker(metrics, 100):
                        self.send(chunk)
                    logger.info("%d metrics sent", len(metrics))
                time.sleep(10)
        except KeyboardInterrupt:
            logger.info("Reporting has stopped by the user. Exiting...")
        finally:
            self.disconnect()
