# coding=utf-8

import logging
import uuid

from django.contrib.auth.models import User
from django.db.models import (Model, BooleanField, CharField, DateTimeField,
                              ForeignKey, TextField)
from django.utils import timezone
from django.utils.translation import ugettext_lazy as _
from model_utils.models import TimeStampedModel
from sizefield.models import FileSizeField

from .tasks import local_tasks, remote_tasks

logger = logging.getLogger(__name__)


class DataStore(Model):

    """Collection of virtual disks.
    """
    name = CharField(max_length=100, unique=True, verbose_name=_('name'))
    path = CharField(max_length=200, unique=True, verbose_name=_('path'))
    hostname = CharField(max_length=40, unique=True,
                         verbose_name=_('hostname'))

    class Meta:
        ordering = ['name']
        verbose_name = _('datastore')
        verbose_name_plural = _('datastores')

    def __unicode__(self):
        return u'%s (%s)' % (self.name, self.path)


class Disk(TimeStampedModel):

    """A virtual disk.
    """
    TYPES = [('qcow2-norm', 'qcow2 normal'), ('qcow2-snap', 'qcow2 snapshot'),
             ('iso', 'iso'), ('raw-ro', 'raw read-only'), ('raw-rw', 'raw')]
    name = CharField(blank=True, max_length=100, verbose_name=_("name"))
    filename = CharField(max_length=256, verbose_name=_("filename"))
    datastore = ForeignKey(DataStore, verbose_name=_("datastore"),
                           help_text=_("The datastore that holds the disk."))
    type = CharField(max_length=10, choices=TYPES)
    size = FileSizeField()
    base = ForeignKey('self', blank=True, null=True,
                      related_name='derivatives')
    ready = BooleanField(default=False)
    dev_num = CharField(default='a', max_length=1,
                        verbose_name=_("device number"))
    destroyed = DateTimeField(blank=True, default=None, null=True)

    class Meta:
        ordering = ['name']
        verbose_name = _('disk')
        verbose_name_plural = _('disks')

    class WrongDiskTypeError(Exception):
        def __init__(self, type):
            self.type = type

        def __str__(self):
            return ("Operation can't be invoked on a disk of type '%s'." %
                    self.type)

    class DiskInUseError(Exception):
        def __init__(self, disk):
            self.disk = disk

        def __str__(self):
            return ("The requested operation can't be performed on disk "
                    "'%s (%s)' because it is in use." %
                    (self.disk.name, self.disk.filename))

    @property
    def path(self):
        return self.datastore.path + '/' + self.filename

    @property
    def format(self):
        return {
            'qcow2-norm': 'qcow2',
            'qcow2-snap': 'qcow2',
            'iso': 'iso',
            'raw-ro': 'raw',
            'raw-rw': 'raw',
        }[self.type]

    @property
    def device_type(self):
        return {
            'qcow2': 'vd',
            'raw': 'vd',
            'iso': 'hd',
        }[self.format]

    def is_in_use(self):
        return self.instance_set.exclude(state='SHUTOFF').exists()

    def get_exclusive(self):
        """Get an instance of the disk for exclusive usage.

        This method manipulates the database only.
        """
        type_mapping = {
            'qcow2-norm': 'qcow2-snap',
            'iso': 'iso',
            'raw-ro': 'raw-rw',
        }

        if self.type not in type_mapping.keys():
            raise self.WrongDiskTypeError(self.type)

        filename = self.filename if self.type == 'iso' else str(uuid.uuid4())
        new_type = type_mapping[self.type]

        return Disk.objects.create(base=self, datastore=self.datastore,
                                   filename=filename, name=self.name,
                                   size=self.size, type=new_type)

    def get_vmdisk_desc(self):
        return {
            'source': self.path,
            'driver_type': self.format,
            'driver_cache': 'default',
            'target_device': self.device_type + self.dev_num
        }

    def get_disk_desc(self):
        return {
            'name': self.filename,
            'dir': self.datastore.path,
            'format': self.format,
            'size': self.size,
            'base_name': self.base.filename if self.base else None,
            'type': 'snapshot' if self.type == 'qcow2-snap' else 'normal'
        }

    def __unicode__(self):
        return u"%s (#%d)" % (self.name, self.id)

    def clean(self, *args, **kwargs):
        if self.size == "" and self.base:
            self.size = self.base.size
        super(Disk, self).clean(*args, **kwargs)

    def deploy(self, user=None, task_uuid=None):
        """Reify the disk model on the associated data store.

        :param self: the disk model to reify
        :type self: storage.models.Disk

        :return: True if a new reification of the disk has been created;
                 otherwise, False.
        :rtype: bool
        """
        if self.ready:
            return False

        act = DiskActivity(activity_code='storage.Disk.deploy')
        act.disk = self
        act.started = timezone.now()
        act.state = 'PENDING'
        act.task_uuid = task_uuid
        act.user = user
        act.save()

        # Delegate create / snapshot jobs
        queue_name = self.datastore.hostname + ".storage"
        disk_desc = self.get_disk_desc()
        if self.type == 'qcow2-snap':
            act.update_state('CREATING SNAPSHOT')
            remote_tasks.snapshot.apply_async(args=[disk_desc],
                                              queue=queue_name).get()
        else:
            act.update_state('CREATING DISK')
            remote_tasks.create.apply_async(args=[disk_desc],
                                            queue=queue_name).get()

        self.ready = True
        self.save()

        act.finish('SUCCESS')
        return True

    def deploy_async(self, user=None):
        """Execute deploy asynchronously.
        """
        local_tasks.deploy.apply_async(args=[self, user],
                                       queue="localhost.man")

    def destroy(self, user=None, task_uuid=None):
        # TODO add activity logging
        self.destroyed = timezone.now()
        self.save()

    def destroy_async(self, user=None):
        local_tasks.destroy.apply_async(args=[self, user],
                                        queue='localhost.man')

    def restore(self, user=None, task_uuid=None):
        """Restore destroyed disk.
        """
        # TODO
        pass

    def restore_async(self, user=None):
        local_tasks.restore.apply_async(args=[self, user],
                                        queue='localhost.man')

    def save_as(self, name, user=None, task_uuid=None):
        mapping = {
            'qcow2-snap': ('qcow2-norm', self.base),
        }
        if self.type not in mapping.keys():
            raise self.WrongDiskTypeError(self.type)

        if self.is_in_use():
            raise self.DiskInUseError(self)

        # from this point on, the caller has to guarantee that the disk is not
        # going to be used until the operation is complete

        act = DiskActivity(activity_code='storage.Disk.save_as')
        act.disk = self
        act.started = timezone.now()
        act.state = 'PENDING'
        act.task_uuid = task_uuid
        act.user = user
        act.save()

        filename = str(uuid.uuid4())
        new_type, new_base = mapping[self.type]

        disk = Disk.objects.create(base=new_base, datastore=self.datastore,
                                   filename=filename, name=name,
                                   size=self.size, type=new_type)

        queue_name = self.datastore.hostname + ".storage"
        remote_tasks.merge.apply_async(args=[self.get_disk_desc(),
                                             disk.get_disk_desc()],
                                       queue=queue_name).get()

        disk.ready = True
        disk.save()
        act.finish('SUCCESS')

        return disk


class DiskActivity(TimeStampedModel):
    activity_code = CharField(verbose_name=_('activity_code'), max_length=100)
    task_uuid = CharField(verbose_name=_('task_uuid'), blank=True,
                          max_length=50, null=True, unique=True)
    disk = ForeignKey(Disk, verbose_name=_('disk'),
                      related_name='activity_log')
    user = ForeignKey(User, verbose_name=_('user'), blank=True, null=True)
    started = DateTimeField(verbose_name=_('started'), blank=True, null=True)
    finished = DateTimeField(verbose_name=_('finished'), blank=True, null=True)
    result = TextField(verbose_name=_('result'), blank=True, null=True)
    state = CharField(verbose_name=_('state'), default='PENDING',
                      max_length=50)

    def update_state(self, new_state):
        self.state = new_state
        self.save()

    def finish(self, result=None):
        if not self.finished:
            self.finished = timezone.now()
            self.result = result
            self.state = 'COMPLETED'
            self.save()
