#include "mainwindow.h"
#include "ui_mainwindow.h"
#include "sshdialog.h"

///Implementation for class MainWindow
///
///---------------------------------------------------------------------------------------------------
/// Basic Methods

/*!
 * \brief Constructor, sets up ui, disable tabs other than login
 * \param parent
 */
MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);
    ui->tabWidget->setTabEnabled(1, false);
    ui->tabWidget->setTabEnabled(2, false);
}

/*!
 * \brief Destructor
 */
MainWindow::~MainWindow()
{
    delete ui;
}

///---------------------------------------------------------------------------------------------------
/// Methods for loading data to the UI

/*!
 * \brief Adds a VM to the listWidget, sets its id, connects all Quick Action buttons
 * \param id
 * \param name
 */
void MainWindow::addVM(std::string id, std::string name){
    QListWidgetItem* item = new QListWidgetItem(ui->listWidget);
    ComputeWidget* cw = new ComputeWidget();
    cw->setID(id);
    cw->setName(name);
    connect(this, SIGNAL(stateChanged(std::string, std::string)), cw, SLOT(stateChangedHandler(std::string, std::string)));
    emit stateChanged(myController.getState(id), id);
    connect(cw, SIGNAL(renew(std::string)), this, SLOT(renewClicked(std::string)));
    connect(cw, SIGNAL(startshutdown(std::string)), this, SLOT(startshutdownClicked(std::string)));
    connect(cw, SIGNAL(wakeupsleep(std::string)), this, SLOT(wakeupsleepClicked(std::string)));
    item->setSizeHint(cw->minimumSize());
    ui->listWidget->addItem(item);
    ui->listWidget->setItemWidget(item, cw);
}

void MainWindow::addTemplate(std::string id, std::string name){
     QListWidgetItem* item = new QListWidgetItem(ui->lwTemplates);
     TemplateWidget* tw = new TemplateWidget();
     tw->setId(id);
     tw->setName(name);
     //connect
     item->setSizeHint(tw->minimumSize());
     ui->lwTemplates->addItem(item);
     ui->lwTemplates->setItemWidget(item, tw);
}

/*!
 * \brief loads all data to the UI, which are stored in Controller, DOES NOT COMMUNICATE WITH THE CLOUD SERVER!!
 */
void MainWindow::loadAllData(){
    VM_names = myController.getVMs();
    ui->listWidget->clear();
    for(size_t i = 0; i < VM_names.size(); i++)
        addVM(std::to_string(i), VM_names.at(i));
    ui->lwTemplates->clear();
    auto templateNames = myController.getTemplates();
    for(size_t i = 0; i < templateNames.size(); i++)
        addTemplate(std::to_string(i), templateNames.at(i));
}

/*!
 * \brief Loads all data for a VM specified by the id, does NOT communicate with the cloud server!!
 * \param id
 */
void MainWindow::loadVMData(std::string id){
    ui->lStatus->setText(myController.getState(id).c_str());
    ComputeData data = myController.getDataByID(id.c_str());
    currentSelectedVM = data;
    ui->lVMName->setText(data.name.c_str());
    //--- for displaying initial hint
    QFont font = ui->lVMName->font();
    font.setPointSize(20);
    font.setBold(false);
    ui->lVMName->setFont(font);
    //---
    ui->lArch->setText(data.architecture.c_str());
    ui->lCPU->setText(std::to_string(data.cpu).c_str());
    ui->lMemory->setText(std::to_string(data.memory).c_str());
    ui->lCores->setText(std::to_string(data.cores).c_str());
    ui->lHostname->setText((data.hostname + ":" + std::to_string(data.port)).c_str());
    ui->lProtocol->setText(data.protocol.c_str());
    ui->lUsernameVM->setText(data.username.c_str());
    ui->lPasswordVM->setText(data.password.c_str());
    ui->lSuspended->setText(data.leaseSuspend.c_str());
    ui->lRemoved->setText(data.leaseRemoved.c_str());
}


///---------------------------------------------------------------------------------------------------
/// Button handlers

///Tab 0 - Login Screen

/*!
 * \brief Login Button handler, connects to server, if authentication is succesful enables all tabs, and loads all data
 */
void MainWindow::on_bLogin_clicked()
{   try{
        std::string serverUrl = parseConfig();//"https://vm.ik.bme.hu:9366/occi/";
        myController.connect(serverUrl);
        std::string username = ui->iUsername->text().toStdString();
        std::string password = ui->iPassword->text().toStdString();
        if(!myController.login(username, password)){
            ui->lInfo->setText("Invalid username or password...");
            return;
        }
        ui->tabWidget->setTabEnabled(1, true);
        ui->tabWidget->setTabEnabled(2, true);
        loadAllData();
        ui->tabWidget->setCurrentIndex(1);
    }
    catch(std::logic_error e){
        QMessageBox msgBox;
        msgBox.setText(e.what());
        msgBox.exec();
    }
}

std::string MainWindow::parseConfig(){
    std::string line;
    std::ifstream myfile ("config.txt");
    if (myfile.is_open())
    {
      while ( getline (myfile,line) )
      {
        return line;
      }
      myfile.close();
    }
    else
        throw std::logic_error("Unable to configure client. Please add a config.txt with the cloud server's url!");
    return "";
}

/// Tab 1 - Virtual Machines

/*!
 * \brief handles Renew button, all static data is refreshed
 * \param id
 */
void MainWindow::renewClicked(std::string id){
    myController.renewByID(id);
    loadVMData(id);
    ui->listWidget->item(std::stoi(id))->setSelected(true);
}

/*!
 * \brief handles Start/Shutdown button (fetches data from server)
 * \param id
 */
void MainWindow::startshutdownClicked(std::string id){
    try{
        std::string state = myController.getState(id);
        if(state == "inactive"){
            myController.startByID(id);
        }
        else {
            myController.shutdownByID(id);
        }
        emit stateChanged(myController.getState(id), id);
        loadVMData(id);
        ui->listWidget->item(std::stoi(id))->setSelected(true);
    }
    catch(std::logic_error e){
        showErrorMessage(e.what());
    }
}

/*!
 * \brief Handles the Wakeup/Sleep toggle button (fetches data from server)
 * \param id
 */
void MainWindow::wakeupsleepClicked(std::string id){
    try{
        std::string state = myController.getState(id);
        if(state == "active"){ //sleep
            myController.sleepByID(id);
        }
        else if(state == "suspended" || state == "inactive"){ //wake up
            myController.wakeupByID(id);
        }
        else{
            throw std::runtime_error("undefined VM STATE!!!");
        }
        emit stateChanged(myController.getState(id), id);
        loadVMData(id);
        ui->listWidget->item(std::stoi(id))->setSelected(true);
    }
    catch(std::logic_error e){
        showErrorMessage(e.what());
    }
}

void MainWindow::showErrorMessage(std::string errorMsg){
    QMessageBox msgBox;
    msgBox.setWindowTitle("Error!");
    msgBox.setText(errorMsg.c_str());
    msgBox.exec();
}

/*!
 * \brief handles Renew button (fetches data from server)
 * \param id
 */
void MainWindow::on_listWidget_clicked(const QModelIndex &index)
{
    loadVMData(std::to_string(index.row()));
}

/*!
 * \brief handles Refresh button, refreshes Controller's data and loads to the UI  (fetches data from server)
 */
void MainWindow::on_bRefresh_clicked()
{
    myController.refresh();
    loadAllData();
}

/*!
 * \brief handles new VM button, Opens a dialog for choosing from templates
 */
void MainWindow::on_bNewVM_clicked()
{
    VMChooserDialog* v = new VMChooserDialog(this);
    v->setWindowTitle("New Virtual Machine");
    v->set_templates(OcciClient::Template::getTemplates(myController.session));
    connect(v, SIGNAL(send_chosen_template(std::shared_ptr<OcciClient::Template>)), this, SLOT(receive_chosen_template(std::shared_ptr<OcciClient::Template>)));
    v->exec();
}

/*!
 * \brief Slot for dialog, creates new VM by the chosen template
 * \param t
 */
void MainWindow::receive_chosen_template(std::shared_ptr<OcciClient::Template> t){
    myController.newCompute(t);
    loadAllData();
}

/*!
 * \brief handler, shows/hides VM password
 */
void MainWindow::on_bShowPassword_clicked()
{
    if(ui->lPasswordVM->echoMode() == QLineEdit::Password){
        ui->lPasswordVM->setEchoMode(QLineEdit::Normal);
        ui->bShowPassword->setText("Hide");
    }
    else if(ui->lPasswordVM->echoMode() == QLineEdit::Normal){
        ui->lPasswordVM->setEchoMode(QLineEdit::Password);
        ui->bShowPassword->setText("Show");
    }
    else
        ui->lPasswordVM->setEchoMode(QLineEdit::Password);
}

void MainWindow::on_bConnect_clicked()
{
    QTermWidget* console = new QTermWidget(0);
    console->setShellProgram("/usr/bin/sshpass");
    QString connString = currentSelectedVM.connString.c_str();
    QStringList args = connString.split(" ");
    console->setArgs(args);

    console->startShellProgram();

    SSHDialog* sshdialog = new SSHDialog();
    sshdialog->setWindowTitle((currentSelectedVM.hostname + " " + std::to_string(currentSelectedVM.port)).c_str());
    sshdialog->setSSHConsole(console);
    sshdialog->show();
}
