import React, { Component } from 'react'
import { Table, Icon, Menu } from 'semantic-ui-react'
import moment from 'moment'
import _ from 'underscore'

export default class DataTable extends Component {
  constructor(props) {
    super(props)
    this.state = {
      column: null,
      data: this.props.data,
      direction: null,
    }
  }

  handleSort(clickedColumn) {
    const { column, data, direction } = this.state

    if (column !== clickedColumn) {
      this.setState({
        column: clickedColumn,
        data: { ...data, results: _.sortBy(data.results, clickedColumn) },
        direction: 'ascending',
      })
    } else {
      this.setState({
        data: { results: data.results.reverse() },
        direction: direction === 'ascending' ? 'descending' : 'ascending',
      })
    }
  }

  handlePagination(url) {
    this.props.onPagination(url).then(data => this.setState(data))
  }

  handleRowClick(item) {
    const { options } = this.props
    this.props.history.push(`${this.props.match.path}/edit`, { item, options })
  }

  renderField(item, field) {
    const { POST } = this.props.options.actions
    if (item[field] !== null) {
      switch (POST[field].type) {
        case 'datetime':
          return moment(item[field]).format('YYYY.MM.DD hh:mm')
        case 'boolean':
          if (item[field] === true) {
            return <Icon color='green' name='checkmark' size='large' />
          }
          return <Icon color='red' name='close' size='large' />
        case 'field':
        case 'choices':
        case 'integer':
        case 'string':
        default:
          return item[field]
      }
    }

    return ''
  }

  renderItems() {
    const { fields } = this.props
    const { data } = this.state
    return data.results.map(item => (
      <Table.Row key={item.url} onClick={() => this.handleRowClick(item)}>
        {fields.map(field => (
          <Table.Cell key={field}>{this.renderField(item, field)}</Table.Cell>
        ))}
      </Table.Row>
    ))
  }

  render() {
    if (this.props.options !== undefined) {
      const { options: { actions: { POST } }, data, fields } = this.props
      const { column, direction } = this.state

      return (
        <Table sortable selectable singleLine>
          <Table.Header>
            <Table.Row>
              {fields.map(field => (
                <Table.HeaderCell
                  key={field}
                  sorted={column === field ? direction : null}
                  onClick={() => this.handleSort(field)}
                >
                  {POST[field].label}
                </Table.HeaderCell>
              ))}
            </Table.Row>
          </Table.Header>

          <Table.Body>
            { this.renderItems() }
          </Table.Body>

          <Table.Footer>
            <Table.Row>
              <Table.HeaderCell colSpan={fields.length}>
                { data.previous || data.next ?
                  <Menu floated='right' pagination>
                    { data.previous ?
                      <Menu.Item onClick={() => this.handlePagination(data.previous)}>Previous</Menu.Item> : ''
                    }
                    { data.next ?
                      <Menu.Item onClick={() => this.handlePagination(data.next)}>Next</Menu.Item> : ''
                    }
                  </Menu> : ''
                }
              </Table.HeaderCell>
            </Table.Row>
          </Table.Footer>
        </Table>
      )
    }

    return <div />
  }
}
