#include "saml2_ecp.h"

#include <restclient-cpp/connection.h>
#include <lasso/lasso.h>
#include <lasso/xml/soap-1.1/xml_soap11.h>

#include <iostream>
#include <fstream>
#include <string>


using namespace RestClient;


namespace OcciClient{


static inline void check_error(int error_code){
        if(error_code < 0)
            throw std::runtime_error(lasso_strerror(error_code));
}


static inline void check_http_error(Response resp){

    if(resp.code < 400) return;
    LassoSoapFault* fault = lasso_soap_fault_new_from_message(resp.body.c_str());
    const char* message = fault->faultstring;
    if(message == NULL) message = resp.body.c_str();
    throw HttpError(message, resp.code);

}


// SAML2 Profile for ECP (Section 4.2) defines these steps for an ECP
// transaction
// 
// 1. ECP issues HTTP Request to SP
// 2. SP issues <AuthnRequest> to ECP using PAOS
// 3. ECP determines IdP
// 4. ECP conveys <AuthnRequest> to IdP using SOAP
// 5. IdP identifies principal
// 6. IdP issues <Response> to ECP, targeted at SP using SOAP
// 7. ECP conveys <Response> to SP using PAOS
// 8. SP grants or denies access to principal

RestClient::Response saml2_ecp_login(std::string username,
                                     std::string password,
                                     std::string login_url,
                                     std::string metadata,
                                     bool insecure){
    lasso_init();

    auto con = Connection("");
    Response r;

    if (insecure) {
        con.SetHostVerify(false);
        con.SetPeerVerify(false);
    }

    // init ECP session
    LassoServer* ecpContext = lasso_server_new(NULL, NULL, NULL, NULL);
    if(ecpContext == NULL)
        throw std::runtime_error("Cannot create ECP client (ecpContext)");
    lasso_provider_set_protocol_conformance(LASSO_PROVIDER(ecpContext),
                                            LASSO_PROTOCOL_SAML_2_0);
    check_error(lasso_server_add_provider(ecpContext, LASSO_PROVIDER_ROLE_IDP,
                                          metadata.c_str(), NULL, NULL));

    LassoEcp* ecp = lasso_ecp_new(ecpContext);
    if(ecp == NULL)
        throw std::runtime_error("Cannot create ECP client (ecp)");

    try{
        // phase 1.
        HeaderFields headers;
        headers["Accept"] = MEDIA_TYPE_PAOS;
        headers["PAOS"] = PAOS_HEADER;  
        con.SetHeaders(headers);
        r = con.get(login_url);
        check_http_error(r);

        // phase 2-6.
        check_error(lasso_ecp_process_authn_request_msg(ecp, r.body.c_str()));
        con.SetBasicAuth(username, password);
        headers["Content-Type"] = MEDIA_TYPE_SOAP;
        con.SetHeaders(headers);
        LassoProfile* profile = LASSO_PROFILE(ecp);
        r = con.post(profile->msg_url, profile->msg_body);
        check_http_error(r);

        // phase 7-8.
        check_error(lasso_ecp_process_response_msg(ecp, r.body.c_str()));
        headers = HeaderFields();
        headers["Content-Type"] = MEDIA_TYPE_PAOS;
        con.SetHeaders(headers);
        r = con.post(profile->msg_url, profile->msg_body);
        check_http_error(r);
    }
    catch(std::exception&){
        // destroy session
        lasso_server_destroy(ecpContext);
        lasso_ecp_destroy(ecp);
        lasso_shutdown();
        throw;
    }
    // destroy session
    lasso_server_destroy(ecpContext);
    lasso_ecp_destroy(ecp);
    lasso_shutdown();

    return r;

}


}
