// Copyright 2017 Budapest University of Technology and Economics (BME IK)
//
// This file is part of CIRCLE Cloud.
//
// CIRCLE is free software: you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free
// Software Foundation, either version 3 of the License, or (at your option)
// any later version.
//
// CIRCLE is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with CIRCLE.  If not, see <http://www.gnu.org/licenses/>.

#include <string>
#include <exception>
#include <stdexcept>
#include "restclient-cpp/connection.h"
#include "restclient-cpp/restclient.h"
#include "json.hpp"

#include "occisession.h"
#include "saml2_ecp.h"

#include <gq/Document.h>
#include <gq/Node.h>

using std::string;
using nlohmann::json;
using namespace OcciClient;

void OcciSession::setCsrfTokenHeader(){
    auto resp = this->connection->getLastResponse();
    string csrftoken;
    try {
        csrftoken = resp.cookies.at("csrftoken");
    }
    catch (const std::out_of_range& e) {
        return;
    }
    this->connection->AppendHeader("X-CSRFToken", csrftoken);
}

OcciSession::OcciSession(const char* url, bool insecure, bool csrf):url(url){
    RestClient::init();
    this->connection = new RestClient::Connection(url);
    if (insecure) {
        this->connection->SetHostVerify(false);
        this->connection->SetPeerVerify(false);
    }
    this->csrftokenRequired = csrf;
    RestClient::HeaderFields headers;
    headers["Accept"] = "application/json";
    headers["Content-type"] = "application/json";
    headers["Referer"] = url;
    this->connection->SetHeaders(headers);
}

OcciSession::~OcciSession(){
    RestClient::disable();
}

json OcciSession::doRequest(string uri, RequestType type, json body){
    RestClient::Response r;

    if (this->csrftokenRequired)
        this->setCsrfTokenHeader();

    if (type == RequestType::Post)
        r = connection->post(uri, body.dump());
    else if (type == RequestType::Put)
        r = connection->put(uri, body.dump());
    else if (type == RequestType::Delete)
        r = connection->del(uri);
    else
        r = connection->get(uri);
    json result;
    try {
        result = json::parse(r.body);
    }
    catch (std::invalid_argument& e) {
        result = "{}"_json;
        throw std::domain_error("Didn't get a json response from the OCCI server.");
    }
    try {
        throw std::logic_error(result["error"].get<string>());
    }
    catch (std::domain_error e) {
        return result;
    }
}

json OcciSession::get(string uri) {
    return doRequest(uri, RequestType::Get);
}

json OcciSession::post(string uri, json body) {
    return doRequest(uri, RequestType::Post, body);
}

json OcciSession::put(string uri, json body) {
    return doRequest(uri, RequestType::Put, body);
}

json OcciSession::del(string uri) {
    return doRequest(uri, RequestType::Delete);
}

void OcciSession::circleOcciLogin(string username, string password){
    get("login/");
    string body = "{\"username\": \"" + username + "\", \"password\": \"" + password + "\"}";
    post("login/", json::parse(body));
}

void OcciSession::BMELogin(string username, string password,  bool insecure){
    std::string& sp_url = url;
    auto con = new RestClient::Connection("");
    if (insecure) {
        con->SetHostVerify(false);
        con->SetPeerVerify(false);
    }
    con->FollowRedirects(true);
    auto r = con->get(sp_url + "/saml2/login/");
    auto idp_auth_url = con->GetInfo().lastRequest.effectiveUrl;
    std::string body = "j_username=" + username +
                       "&j_password=" + password;
    con->setCookies(r.cookies);
    r = con->post(idp_auth_url, body);
    CDocument doc;
    doc.parse(r.body.c_str());

    CSelection c = doc.find("input[name=RelayState]");
    auto RelayState = c.nodeAt(0).attribute("value");
    c = doc.find("input[name=SAMLResponse]");
    auto SAMLResponse = c.nodeAt(0).attribute("value");
    RestClient::PostData data;
    data["RelayState"] = RelayState;
    data["SAMLResponse"] = SAMLResponse;
    con->clearCookies();
    r = con->post(sp_url + "/saml2/acs/", data);
    this->connection->setCookies(r.cookies);
}


void OcciSession::saml2EcpLogin(std::string username,
                                std::string password,
                                std::string metadata,
                                bool insecure){

    auto r = saml2_ecp_login(username, password,
                             url + "/saml2/login/",
                             metadata,
                             insecure);
    this->connection->setCookies(r.cookies);

}

json OcciSession::queryInterface(){
    return get("-/");
}
