// Copyright 2017 Budapest University of Technology and Economics (BME IK)
//
// This file is part of CIRCLE Cloud.
//
// CIRCLE is free software: you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free
// Software Foundation, either version 3 of the License, or (at your option)
// any later version.
//
// CIRCLE is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with CIRCLE.  If not, see <http://www.gnu.org/licenses/>.

#include <string>
#include <vector>
#include <memory>
#include "json.hpp"
#include "resource.h"
#include "template.h"

#include "compute.h"

using std::string;
using std::vector;
using nlohmann::json;
using std::shared_ptr;
using std::make_shared;
using namespace OcciClient;


void Compute::parseJsonInstance(json instance) {
    Resource::parseJsonInstance(instance);
    this->state = getAttributeFromRendering<string>(instance, "occi.compute.state");
    this->stateMessage = getAttributeFromRendering<string>(instance, "occi.compute.state.message");
    this->memory = getAttributeFromRendering<double>(instance, "occi.compute.memory");
    this->cpuShare = getAttributeFromRendering<int>(instance, "occi.compute.share");
    this->cores = getAttributeFromRendering<int>(instance, "occi.compute.cores");
    this->hostname = getAttributeFromRendering<string>(instance, "occi.compute.hostname");
    this->architecture = getAttributeFromRendering<string>(instance, "occi.compute.architecture");
    this->credentialProtocol = getAttributeFromRendering<string>(instance, "org.circlecloud.occi.credentials.protocol");
    this->credentialPort = getAttributeFromRendering<int>(instance, "org.circlecloud.occi.credentials.port");
    this->credentialHost = getAttributeFromRendering<string>(instance, "org.circlecloud.occi.credentials.host");
    this->credentialUsername = getAttributeFromRendering<string>(instance, "org.circlecloud.occi.credentials.username");
    this->credentialPassword = getAttributeFromRendering<string>(instance, "org.circlecloud.occi.credentials.password");
    this->credentialCommand = getAttributeFromRendering<string>(instance, "org.circlecloud.occi.credentials.command");
    this->leasetimeSuspend = getAttributeFromRendering<string>(instance, "org.circlecloud.occi.leasetime.suspend");
    this->leasetimeRemove = getAttributeFromRendering<string>(instance, "org.circlecloud.occi.leasetime.remove");
    this->rendering = instance;
}

Compute::Compute(OcciSession* session, json instance) : Resource(session, "compute", instance) {
    parseJsonInstance(instance);
}

void Compute::invokeAction(json actionData) {
    auto inst = Resource::invokeAction(actionData);
    parseJsonInstance(inst);
}

shared_ptr<Compute> Compute::getComputeInstance(OcciSession* session, string id) {
    auto uri = "compute/" + id + "/";
    auto instance = session->get(uri);
    return make_shared<Compute>(Compute(session, instance));
}

shared_ptr<Compute> Compute::createComputeInstance(OcciSession* session, Template* t) {
    auto uri = "compute/1/";
    json data = json::parse("{\"mixins\": [\"" + t->getTemplateIdentifier() + "\"]}");
    auto instance = session->put(uri, data);
    return make_shared<Compute>(Compute(session, instance));
}

vector<shared_ptr<Compute>> Compute::getComputeInstances(OcciSession* session){
    auto result = vector<shared_ptr<Compute>>();
    auto data = session->get("compute/");
    for (auto& computeRendering : data["resources"]) {
        auto computeInstance = make_shared<Compute>(Compute(session, computeRendering));
        result.push_back(computeInstance);
    }
    return result;
}

vector<string> Compute::getComputeInstanceIds(OcciSession* session){
    auto computes = getComputeInstances(session);
    auto ids = vector<string>();
    for (auto& compute : computes) {
        ids.push_back(compute->getId());
    }
    return ids;
}


string Compute::getState(){
    return this->state;
}

string Compute::getStateMessage(){
    return this->stateMessage;
}

double Compute::getMemory(){
    return this->memory;
}

int Compute::getCpuShare(){
    return this->cpuShare;
}

int Compute::getCores(){
    return this->cores;
}

string Compute::getHostname(){
    return this->hostname;
}

string Compute::getArchitecture(){
    return this->architecture;
}

string Compute::getCredentialProtocol(){
    return this->credentialProtocol;
}

int Compute::getCredentialPort(){
    return this->credentialPort;
}

string Compute::getCredentialHost(){
    return this->credentialHost;
}

string Compute::getCredentialPassword(){
    return this->credentialPassword;
}

string Compute::getCredentialUsername(){
    return this->credentialUsername;
}

string Compute::getCredentialCommand(){
    return this->credentialCommand;
}

string Compute::getLeasetimeSuspend(){
    return this->leasetimeSuspend;
}

string Compute::getLeasetimeRemove(){
    return this->leasetimeRemove;
}

void Compute::start() {
    json actionData = {
        {"action", "http://schemas.ogf.org/occi/infrastructure/compute/action#start"}
    };
    invokeAction(actionData);
}

void Compute::wakeup() {
    start();
}

void Compute::sleep() {
    json actionData = {
        {"action", "http://schemas.ogf.org/occi/infrastructure/compute/action#suspend"},
        {"attributes", {
            {"method", "suspend"}
        }}
    };
    invokeAction(actionData);
}

void Compute::reboot() {
    json actionData = {
        {"action", "http://schemas.ogf.org/occi/infrastructure/compute/action#restart"},
        {"attributes", {
            {"method", "warm"}
        }}
    };
    invokeAction(actionData);
}

void Compute::reset() {
    json actionData = {
        {"action", "http://schemas.ogf.org/occi/infrastructure/compute/action#restart"},
        {"attributes", {
            {"method", "cold"}
        }}
    };
    invokeAction(actionData);
}

void Compute::shutdown() {
    json actionData = {
        {"action", "http://schemas.ogf.org/occi/infrastructure/compute/action#stop"},
        {"attributes", {
            {"method", "acpioff"}
        }}
    };
    invokeAction(actionData);
}

void Compute::shutoff() {
    json actionData = {
        {"action", "http://schemas.ogf.org/occi/infrastructure/compute/action#stop"},
        {"attributes", {
            {"method", "poweroff"}
        }}
    };
    invokeAction(actionData);
}

void Compute::renew() {
    json actionData = {
        {"action", "http://circlecloud.org/occi/infrastructure/compute/action#renew"},
    };
    invokeAction(actionData);
}
