// Copyright 2017 Budapest University of Technology and Economics (BME IK)
//
// This file is part of CIRCLE Cloud.
//
// CIRCLE is free software: you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free
// Software Foundation, either version 3 of the License, or (at your option)
// any later version.
//
// CIRCLE is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with CIRCLE.  If not, see <http://www.gnu.org/licenses/>.

#ifndef OCCILIB_COMPUTE_H_
#define OCCILIB_COMPUTE_H_

#include <string>

#include "json.hpp"
#include "resource.h"
#include "template.h"

#include <iostream>

namespace OcciClient {

    using std::string;

    class Compute : public Resource {
        private:

            string state;
            string stateMessage;
            double memory;
            int cpuShare;
            int cores;
            string hostname;
            string architecture;

            string credentialProtocol;
            int credentialPort;
            string credentialHost;
            string credentialPassword;
            string credentialUsername;
            string credentialCommand;

            void parseJsonInstance(nlohmann::json instance) {
                Resource::parseJsonInstance(instance);
                this->state = getAttributeFromRendering<string>(instance, "occi.compute.state");
                this->stateMessage = getAttributeFromRendering<string>(instance, "occi.compute.state.message");
                this->memory = getAttributeFromRendering<double>(instance, "occi.compute.memory");
                this->cpuShare = getAttributeFromRendering<int>(instance, "occi.compute.share");
                this->cores = getAttributeFromRendering<int>(instance, "occi.compute.cores");
                this->hostname = getAttributeFromRendering<string>(instance, "occi.compute.hostname");
                this->architecture = getAttributeFromRendering<string>(instance, "occi.compute.architecture");
                this->credentialProtocol = getAttributeFromRendering<string>(instance, "org.circlecloud.occi.credentials.protocol");
                this->credentialPort = getAttributeFromRendering<int>(instance, "org.circlecloud.occi.credentials.port");
                this->credentialHost = getAttributeFromRendering<string>(instance, "org.circlecloud.occi.credentials.host");
                this->credentialUsername = getAttributeFromRendering<string>(instance, "org.circlecloud.occi.credentials.username");
                this->credentialPassword = getAttributeFromRendering<string>(instance, "org.circlecloud.occi.credentials.password");
                this->credentialCommand = getAttributeFromRendering<string>(instance, "org.circlecloud.occi.credentials.command");
                this->rendering = instance;
            }

            Compute(OcciSession* session, nlohmann::json instance)
                : Resource(session, "compute", instance) {
                parseJsonInstance(instance);
            }

        public:

            static std::shared_ptr<Compute> getComputeInstance(OcciSession* session, string id) {
                auto uri = "compute/" + id + "/";
                auto instance = session->get(uri);
                return std::make_shared<Compute>(Compute(session, instance));
            }

            static std::shared_ptr<Compute> createComputeInstance(OcciSession* session, Template t) {
                auto uri = "compute/1/";
                nlohmann::json data = nlohmann::json::parse("{\"mixins\": [\"" + t + "\"]}");
                auto instance = session->put(uri, data);
                return std::make_shared<Compute>(Compute(session, instance));
            }

            string getState(){
                return this->state;
            }

            string getStateMessage(){
                return this->stateMessage;
            }

            double getMemory(){
                return this->memory;
            }

            int getCpuShare(){
                return this->cpuShare;
            }

            int getCores(){
                return this->cores;
            }

            string getHostname(){
                return this->hostname;
            }

            string getArchitecture(){
                return this->architecture;
            }

            string getCredentialProtocol(){
                return this->credentialProtocol;
            }

            int getCredentialPort(){
                return this->credentialPort;
            }

            string getCredentialHost(){
                return this->credentialHost;
            }

            string getCredentialPassword(){
                return this->credentialPassword;
            }

            string getCredentialUsername(){
                return this->credentialUsername;
            }

            string getCredentialCommand(){
                return this->credentialCommand;
            }

            void invokeAction(nlohmann::json actionData) {
                auto inst = Resource::invokeAction(actionData);
                parseJsonInstance(inst);
            }

            void start() {
                nlohmann::json actionData = {
                    {"action", "http://schemas.ogf.org/occi/infrastructure/compute/action#start"}
                };
                invokeAction(actionData);
            }

            void wakeup() {
                start();
            }

            void sleep() {
                nlohmann::json actionData = {
                    {"action", "http://schemas.ogf.org/occi/infrastructure/compute/action#suspend"},
                    {"attributes", {
                        {"method", "suspend"}
                    }}
                };
                invokeAction(actionData);
            }

            void reboot() {
                nlohmann::json actionData = {
                    {"action", "http://schemas.ogf.org/occi/infrastructure/compute/action#restart"},
                    {"attributes", {
                        {"method", "warm"}
                    }}
                };
                invokeAction(actionData);
            }

            void reset() {
                nlohmann::json actionData = {
                    {"action", "http://schemas.ogf.org/occi/infrastructure/compute/action#restart"},
                    {"attributes", {
                        {"method", "cold"}
                    }}
                };
                invokeAction(actionData);
            }

            void shutdown() {
                nlohmann::json actionData = {
                    {"action", "http://schemas.ogf.org/occi/infrastructure/compute/action#stop"},
                    {"attributes", {
                        {"method", "acpioff"}
                    }}
                };
                invokeAction(actionData);
            }

            void shutoff() {
                nlohmann::json actionData = {
                    {"action", "http://schemas.ogf.org/occi/infrastructure/compute/action#stop"},
                    {"attributes", {
                        {"method", "poweroff"}
                    }}
                };
                invokeAction(actionData);
            }
    };

}

#endif // OCCILIB_COMPUTE_H
