#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
Main program of the Client written for CIRCLE Cloud.
The Client job is to help the ease of use of the cloud system.
"""

import argparse
import logging
import platform
import tempfile
from time import gmtime, strftime

class Struct:
    """
    A struct used for parameter passing

    Keyword arguments:
    state    -- State of the Virtual Computer (running, etc..)
    protocol -- SSH and RDP possible
    host     -- Address of the Virtual Computer
    port     -- The port where we can access the Virtual Computer
    user     -- Username used for the connection
    password -- Password used for the connection
    """
    pass


def parse_arguments():
    """
    Argument parser, based on the argparse module

    Keyword arguments:
    @return args -- arguments given by console
    """
    parser = argparse.ArgumentParser()
    parser.add_argument(
        "uri", type=str, help="Specific schema handler", nargs='?',
        default=None)
    parser.add_argument(
        "-l", "--loglevel",
        help="The level of logging severity", type=str,
        choices=['DEBUG', 'INFO', 'WARNING', 'ERROR', 'CRITICAL'],
        default="INFO")
    parser.add_argument(
        "-f", "--logfile",
        help="Explicit location of the wanted logfile location and name",
        type=str,
        default=("%(directory)s\\%(file)s" % {
            'directory': tempfile.gettempdir(),
            'file': 'circle_client.log'}))
    args = parser.parse_args()
    return args


def main():
    """
    Main program
    """
    args = parse_arguments()
    logging.basicConfig(filename='%s' % args.logfile,
                        format='%(levelname)s:%(message)s')
    logger = logging.getLogger()
    logger.setLevel(args.loglevel)
    logger.info("----------------------------")
    logger.info("Client started running at %s",
                strftime("%Y-%d-%m %H:%M:%S GMT", gmtime()))
    try:

        if args.uri is not None:
            vm = Struct()
            x, vm.protocol, vm.user, vm.password, vm.host, vm.port = \
                args.uri.split(':', 5)
            logger.info(("Received the following URI: %(handler)s:"
                         "%(protocol)s:%(user)s:password:%(host)s:%(port)s" % {
                             'handler': x,
                             'protocol': vm.protocol,
                             'user': vm.user,
                             'host': vm.host,
                             'port': vm.port
                         }))
            vm.protocol = vm.protocol.upper()
            logger.debug("The Client split it followingly: protocol -> %s | "
                         "user -> %s | password -> %s | "
                         "host -> %s | port -> %s",
                         vm.protocol, vm.user, vm.password, vm.host, vm.port)
        else:
            logger.critical("Client did not receive an URI which would be "
                            "necessary to continue")
        if platform.system() == "Windows":
            logger.debug('Windows OS found, proceeding to connect methods')
            from cloud_connect_from_windows import connect
        connect(vm)
    except:
        logger.exception("An exception was raised before connect methods"
                         "could be invoked")
        print("Unknown error occurred! Please contact the developers!")


if __name__ == "__main__":
    main()
