import netifaces
from netaddr import IPNetwork
import logging
import subprocess
import os
import os.path


logger = logging.getLogger()

IF_CONF_PREFIX = '/etc/hostname.'


def get_interfaces_openbsd(interfaces):
    for ifname in netifaces.interfaces():
        if ifname == 'lo0':
            continue
        logger.debug("get_interfaces: " + ifname)
        netif = netifaces.ifaddresses(ifname)
        conf = None
        if netif:
            mac_list = netif.get(netifaces.AF_LINK)
            if mac_list:
                mac = mac_list[0]['addr']
                logger.debug("get_interfaces: " + mac)
                conf = interfaces.get(mac.upper())
        if conf:
            yield ifname, conf


def remove_interfaces_openbsd(devices):
    for device in devices:
        if_file = IF_CONF_PREFIX + device
        subprocess.call(('/sbin/ifconfig', device, '-inet', '-inet6'))
        subprocess.call(('/sbin/ifconfig', device, 'destroy'))
        if os.path.isfile(if_file):
            logger.debug("remove interface configuration: " + if_file)
            os.unlink(if_file)
        else:
            logger.debug(
                "unable to remove interface configuration: " + if_file)


def change_ip_openbsd(interfaces, nameservers):
    data = list(get_interfaces_openbsd(interfaces))
    remove_interfaces_openbsd(dict(data).keys())

    # Set IPs
    for device, conf in data:
        if_file = IF_CONF_PREFIX + device
        with open(if_file, 'w') as f:
            ipv4_alias_counter = ipv6_alias_counter = 0
            for i in conf['addresses']:
                ip = IPNetwork(i)
                alias = ''
                typ = 'inet' if ip.version == 4 else 'inet6'
                if ip.version == 6 and ipv6_alias_counter > 0:
                    alias = 'alias'
                    ipv6_alias_counter += 1
                if ip.version == 4 and ipv4_alias_counter > 0:
                    alias = 'alias'
                    ipv4_alias_counter += 1
                f.write('%(type)s %(ip)s %(alias)s\n'
                        % {
                            'type': typ,
                            'ip': ip,
                            'alias': alias
                        })
        os.chmod(if_file, 0540)

    # Set default gateway from last config
    if len(data) > 0:
        device, conf = data[len(data) - 1]
        with open('/etc/mygate', 'w') as f:
            f.write(conf['gw4'] + '\n')
            f.write(conf['gw6'] + '\n')

    # Set nameservers
    with open('/etc/resolv.conf', 'w') as f:
        for ns in nameservers:
            f.write('nameserver %s\n' % ns)

    # Restart network
    subprocess.call(('/bin/sh', '/etc/netstart'))


# example:
# change_ip_ubuntu({
#    u'02:00:00:02:A3:E8': {
#        u'gw4': u'10.1.0.254', 'gw6': '2001::ffff',
#        u'addresses': [u'10.1.0.84/24', '10.1.0.1/24', '2001::1/48']},
#    u'02:00:00:02:A3:E9': {
#        u'gw4': u'10.255.255.1', u'addresses': [u'10.255.255.9']}},
#    '8.8.8.8')
