#!/usr/bin/env python
# -*- coding: utf-8 -*-

from os import mkdir, environ, chmod
from shutil import rmtree, move
import subprocess
import sys
import pwd
import logging
import fileinput
import tarfile
from os.path import expanduser, join, exists
from glob import glob
from StringIO import StringIO
from base64 import decodestring
from hashlib import md5


from ssh import PubKey
from .network import change_ip_openbsd
from context import BaseContext

from twisted.internet import reactor
import jinja2


logging.basicConfig()
logger = logging.getLogger()
level = environ.get('LOGLEVEL', 'INFO')
logger.setLevel(level)

SSH_DIR = expanduser('~cloud/.ssh')
AUTHORIZED_KEYS = join(SSH_DIR, 'authorized_keys')

STORE_DIR = '/store'

USMB_CONF = '/etc/circle/usmb.conf'

MOUNT_TEMPLATE_OPENBSD = "none /store circle-userstore rw 0 0\n"


def render_template_to_file(template_name, filename, context):
    with open(filename, 'w') as f:
        with open(template_name) as tf:
            template = jinja2.Template(''.join(tf.readlines()))
            template.stream(context).dump(f)


class Context(BaseContext):

    # http://stackoverflow.com/questions/12081310/
    # python-module-to-change-system-date-and-time
    @staticmethod
    def _openbsd_set_time(time):
        import ctypes
        import ctypes.util

        CLOCK_REALTIME = 0

        class timespec(ctypes.Structure):
            _fields_ = [('tv_sec', ctypes.c_long),
                        ('tv_nsec', ctypes.c_long)]

        libc = ctypes.CDLL(ctypes.util.find_library('libc'))

        ts = timespec()
        ts.tv_sec = int(time)
        ts.tv_nsec = 0

        libc.clock_settime(CLOCK_REALTIME, ctypes.byref(ts))

    @staticmethod
    def _change_user_password(username, password):
        enc_proc = subprocess.Popen(
            ['/usr/bin/encrypt'],
            stdin=subprocess.PIPE, stdout=subprocess.PIPE)
        enc_password, _ = enc_proc.communicate('%s\n' % password)
        subprocess.Popen(
            ['/usr/sbin/usermod', '-p', enc_password.strip(), username])

    @staticmethod
    def change_password(password):
        Context._change_user_password('cloud', password)
        Context._change_user_password('root', password)

    @staticmethod
    def restart_networking():
        logger.debug("restart_networking")

    @staticmethod
    def change_ip(interfaces, dns):
        nameservers = dns.replace(' ', '').split(',')
        change_ip_openbsd(interfaces, nameservers)

    @staticmethod
    def set_time(time):
        Context._openbsd_set_time(float(time))
        try:
            subprocess.call(['rcctl' 'restart', 'ntpd'])
        except Exception:
            pass

    @staticmethod
    def set_hostname(hostname):
        with open('/etc/myname', 'w') as f:
            f.write(hostname)

        with open('/etc/hosts', 'w') as f:
            f.write("127.0.0.1 localhost\n"
                    "127.0.1.1 %s\n" % hostname)

        subprocess.call(['/bin/hostname', hostname])

    @staticmethod
    def mount_store(host, username, password):
        uid = pwd.getpwnam('cloud').pw_uid
        data = {'host': host, 'username': username,
                'password': password, 'uid': uid}
        data['dir'] = STORE_DIR
        if not exists(STORE_DIR):
            mkdir(STORE_DIR)

        render_template_to_file('openbsd/usmb.conf', USMB_CONF, data)
        chmod(USMB_CONF, 0600)

        for line in fileinput.input('/etc/fstab', inplace=True):
            if not (line.startswith('none') and ' circle-userstore ' in line):
                print line.rstrip()

        with open('/etc/fstab', 'a') as f:
            f.write(MOUNT_TEMPLATE_OPENBSD % data)

        subprocess.call('/sbin/mount /store', shell=True)

    @staticmethod
    def get_keys():
        retval = []
        try:
            with open(AUTHORIZED_KEYS, 'r') as f:
                for line in f.readlines():
                    try:
                        retval.append(PubKey.from_str(line))
                    except Exception:
                        logger.exception(u'Invalid ssh key: ')
        except IOError:
            pass
        return retval

    @staticmethod
    def _save_keys(keys):
        print keys
        try:
            mkdir(SSH_DIR)
        except OSError:
            pass
        with open(AUTHORIZED_KEYS, 'w') as f:
            for key in keys:
                f.write(unicode(key) + '\n')

    @staticmethod
    def add_keys(keys):
        new_keys = Context.get_keys()
        for key in keys:
            try:
                p = PubKey.from_str(key)
                if p not in new_keys:
                    new_keys.append(p)
            except Exception:
                logger.exception(u'Invalid ssh key: ')
        Context._save_keys(new_keys)

    @staticmethod
    def del_keys(keys):
        new_keys = Context.get_keys()
        for key in keys:
            try:
                p = PubKey.from_str(key)
                try:
                    new_keys.remove(p)
                except ValueError:
                    pass
            except Exception:
                logger.exception(u'Invalid ssh key: ')
        Context._save_keys(new_keys)

    @staticmethod
    def cleanup():
        filelist = ([
            '/root/.bash_history',
            '/home/cloud/.bash_history',
            '/root/.ssh',
            '/home/cloud/.ssh',
            '/root/.lesshst',
            '/home/cloud/.lesshst',
            '/root/.history',
            '/home/cloud/.history',
            '/root/.viminfo',
            '/home/cloud/.viminfo',
            USMB_CONF,
            ]
            + glob('/etc/ssh/ssh_host_*'))
        for f in filelist:
            rmtree(f, ignore_errors=True)

        subprocess.call(('/usr/bin/ssh-keygen', '-A'))

    @staticmethod
    def start_access_server():
        subprocess.call(('/usr/sbin/rcctl', 'start', 'sshd'))

    @staticmethod
    def append(data, filename, chunk_number, uuid):
        if chunk_number == 0:
            flag = "w"
        else:
            flag = "a"
        with open(filename, flag) as myfile:
            myfile.write(data)

    @staticmethod
    def update(filename, executable, checksum, uuid):
        working_directory = sys.path[0]
        new_dir = working_directory + '.new'
        old_dir = working_directory + '.old.%s' % uuid
        with open(filename, "r") as f:
            data = f.read()
            local_checksum = md5(data).hexdigest()
            if local_checksum != checksum:
                raise Exception("Checksum missmatch the file is damaged.")
            decoded = StringIO(decodestring(data))
        try:
            tar = tarfile.TarFile.open("dummy", fileobj=decoded, mode='r|gz')
            tar.extractall(new_dir)
        except tarfile.ReadError as e:
            logger.error(e)
        move(working_directory, old_dir)
        move(new_dir, working_directory)
        logger.info("Transfer completed!")
        reactor.stop()

    @staticmethod
    def ipaddresses():
        import netifaces
        args = {}
        interfaces = netifaces.interfaces()
        for i in interfaces:
            if i == 'lo0':
                continue
            args[i] = []
            addresses = netifaces.ifaddresses(i)
            args[i] = ([x['addr']
                        for x in addresses.get(netifaces.AF_INET, [])] +
                       [x['addr']
                        for x in addresses.get(netifaces.AF_INET6, [])
                        if '%' not in x['addr']])
        return args

    @staticmethod
    def get_agent_version():
        try:
            with open('version.txt') as f:
                return f.readline()
        except IOError:
            return None

    @staticmethod
    def send_expiration(url):
        import notify
        notify.notify(url)
