from django.test import TestCase
from django.contrib.auth.models import User
from unittest.mock import patch

from template.models import ImageTemplate
from image.models import Image
from instance.models import Flavor, Lease, Instance

from interface_openstack.interface.vm.resources import Flavor as RemoteFlavor


class TemplateModelsTest(TestCase):
    @staticmethod
    def create_user(name="test_user", email="test_email", password="test_password"):
        return User.objects.create_user(username=name, email=email, password=password)

    @staticmethod
    def create_flavor():
        with patch('instance.models.interface') as mock_interface:
            mock_interface.create_flavor.return_value = RemoteFlavor("123456789",
                                                                     "test_flavor",
                                                                     2, 1, 1)
            return Flavor.create("test_flavor", "test flavor description", 2, 1, 1, 2)

    def create_instance(self):
        params = {"name": "test_instance",
                  "description": "test instance description",
                  "access_method": "ssh",
                  "system": "test_system",
                  "time_of_suspend": "2019-05-28 23:59",
                  "time_of_delete": "2019-05-28 23:59",

                  }
        flavor = self.create_flavor()
        lease = Lease.objects.create(name="test_lease", description="test lease description",
                                     suspend_interval_in_sec=100, delete_interval_in_sec=100)
        owner = self.create_user("test_instance_user", "instance_user", "test_password")
        remote_id = "123456789"
        image = Image.objects.create(name="test_image", description="test image description",
                                     uploaded_by_user=True, remote_id="123456789123456789",
                                     created_by=owner)
        template = ImageTemplate.objects.create(name="test_template",
                                                description="test descripton",
                                                image=image,
                                                lease=lease,
                                                flavor=flavor,
                                                created_by=owner,
                                                type='U')
        return Instance.create(lease=lease, owner=owner, flavor=flavor,
                               remote_id=remote_id, template=template, params=params)

    def test_create_from_instance(self):
        with patch(
                'template.models.Image'
        ) as mock_image:
            user = self.create_user()
            image = Image.objects.create(name="test_image",
                                         description="test image description",
                                         uploaded_by_user=False,
                                         remote_id="123456789",
                                         created_by=user)
            instance = self.create_instance()
            description = "test template description"
            name = "test_template"
            mock_image.create_from_instance()
            mock_image.create_from_instance.return_value = image

            template = ImageTemplate.create_from_instance(name, description, instance, user)

            self.assertEqual(template.name, name)
            self.assertEqual(template.created_by, user)
            self.assertEqual(template.type, "I")
            self.assertEqual(template.description, description)
            self.assertEqual(template.lease, instance.lease)
            self.assertEqual(template.flavor, instance.flavor)
            self.assertIsInstance(template, ImageTemplate)
