from django.db import models
from django.contrib.auth.models import User
from django.conf import settings

from image.models import Disk, Image
from instance.models import Lease, Flavor
from interface_openstack.implementation.storage.openstack_snapshot_manager import SnapshotManager


class BaseTemplate(models.Model):

    """Virtual machine template.
    """

    name = models.CharField(
        max_length=100,
        verbose_name="name",
        help_text="Human readable name of template."
    )
    description = models.TextField(
        verbose_name="description",
        blank=True,
        help_text="Description of the template."
    )
    # remote_id = models.CharField(
    #     max_length=40,
    #     unique=True,
    #     verbose_name="remote_ID",
    #     help_text="ID, which helps access the template."
    # )
    created_at = models.DateTimeField(
        auto_now_add=True,
        editable=False,
        help_text="Date, when the template created."
    )
    created_by = models.ForeignKey(
        User,
        on_delete=models.DO_NOTHING,
        related_name="created_templates",
        help_text="The user, who create the template"
    )
    flavor = models.ForeignKey(Flavor, help_text="Reasources given to the vm",
                               verbose_name="flavor", on_delete="CASCADE",
                               related_name='templates')
    lease = models.ForeignKey(Lease, on_delete="CASCADE",
                              related_name='templates')
    network_id = models.CharField(
        max_length=100,
        verbose_name="network_id",
        help_text="The new instance will be in this network.",
        null=True,
        blank=True
    )


class DiskTemplate(BaseTemplate):
    disk = models.ForeignKey(
        Disk,
        related_name="templates",
        on_delete=models.CASCADE,
        help_text="The disk where the template is located."
    )

    @classmethod
    def create_from_volume(cls, name, description, disk, user):
        interface = SnapshotManager(settings.CONNECTION)
        remote_template = interface.create_from_volume(disk.remote_id)
        remote_id = remote_template.id
        new_template = cls.create(
            name=name,
            description=description,
            disk=disk,
            remote_id=remote_id,
            created_by=user
        )
        return new_template


class ImageTemplate(BaseTemplate):
    TYPES = (
        ('U', 'User create the template from image'),
        ('I', 'Template created from instance'),
        ('P', '"Pure" template'),
    )

    image = models.ForeignKey(
        Image,
        related_name="templates",
        on_delete=models.CASCADE,
        help_text=""
    )
    type = models.CharField(max_length=10, choices=TYPES, default="U")

    @classmethod
    def create(cls, name, description,  image, lease, flavor, created_by, type='U'):
        inst = cls(name=name, description=description, image=image, lease=lease,
                   flavor=flavor, created_by=created_by, type=type)
        inst.full_clean()
        inst.save()
        return inst

    @classmethod
    def create_from_instance(cls, name, description, instance, user):
        image = Image.create_from_instance(user, instance, description)
        new_template = cls.create(
            name=name,
            description=description,
            created_by=user,
            image=image,
            lease=instance.lease,
            flavor=instance.flavor,
            type="I"
        )
        return new_template
