from instance.serializers import InstanceSerializer, FlavorSerializer, LeaseSerializer
from django.http import Http404
from rest_framework.viewsets import ViewSet, ModelViewSet
from rest_framework.response import Response
from rest_framework import status
from rest_framework.decorators import action

from template.serializers import InstanceFromTemplateSerializer
from instance.models import Instance, Flavor, Lease
from template.models import ImageTemplate
from template.serializers import ImageTemplateModelSerializer


class InstanceViewSet(ViewSet):

    def get_object(self, pk):
        try:
            return Instance.objects.get(pk=pk)
        except Instance.DoesNotExist:
            raise Http404

    def list(self, request):
        instances = Instance.objects.all()
        return Response(InstanceSerializer(instances, many=True).data)

    def create(self, request):
        data = request.data
        template = ImageTemplate.objects.get(pk=data["template"])

        # TODO: if the user can select anouther lease and flavor
        #       That will be applied, not from the template

        # flavor = Flavor.objects.get(pk=data["flavor"])
        # lease = Lease.objects.get(pk=data["lease"])

        newInstance = Instance.create_instance_from_template(
            params={"name": data["name"],
                    "description": data["description"],
                    "access_method": data["access"],
                    "system": data["system"],
                    },
            lease=template.lease,
            networks=[{"uuid": template.network_id}],
            template=template,
            flavor=template.flavor,
            owner=request.user,
            disks=None
        )
        return Response(newInstance.pk)

    def retrieve(self, request, pk):
        instance = self.get_object(pk)
        instanceDict = InstanceSerializer(instance).data
        remoteInstance = instance.get_remote_instance()
        remoteInstanceDict = remoteInstance.__dict__

        merged_dict = {**instanceDict, **remoteInstanceDict}

        return Response(merged_dict)

    def update(self, request, pk, format=None):
        instance = self.get_object(pk)
        serializer = InstanceSerializer(instance, data=request.data)
        if serializer.is_valid():
            serializer.save()
            return Response(serializer.data)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    def destroy(self, request, pk, format=None):
        instance = self.get_object(pk)
        instance.delete()
        return Response(status=status.HTTP_204_NO_CONTENT)

    @action(detail=True, methods=["post"])
    def template(self, request, pk):
        instance = self.get_object(pk)
        serializer = InstanceFromTemplateSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        data = serializer.validated_data
        new_template = ImageTemplate.create_from_instance(data["name"],
                                                          data["description"],
                                                          instance,
                                                          request.user)
        serializer = ImageTemplateModelSerializer(instance=new_template)
        return Response(serializer.data)

    @action(detail=True, methods=["POST"])
    def actions(self, request, pk):
        instance = self.get_object(pk)
        success = instance.execute_common_action(action=request.data["action"])

        return Response(success)


class FlavorViewSet(ViewSet):
    """
    Create, update or delete a flavor.
    """

    def get_object(self, pk):
        try:
            return Flavor.objects.get(pk=pk)
        except Flavor.DoesNotExist:
            raise Http404

    def list(self, request, format=None):
        flavors = Flavor.objects.all()
        return Response(FlavorSerializer(flavors, many=True).data)

    def create(self, request, format=None):
        data = request.data
        new_flavor = Flavor.create(name=data["name"],
                                   description=data["description"],
                                   ram=data["ram"],
                                   vcpu=data["vcpu"],
                                   initial_disk=data["initial_disk"],
                                   priority=data["priority"])

        return Response(new_flavor.pk)

    def update(self, request, pk):
        return Response(status=status.HTTP_400_BAD_REQUEST)

    def partial_update(self, request, pk):
        return Response(status=status.HTTP_400_BAD_REQUEST)

    def destroy(self, request, pk):
        flavor = self.get_object(pk)
        flavor.delete()
        return Response(status=status.HTTP_204_NO_CONTENT)


class LeaseViewSet(ModelViewSet):
    queryset = Lease.objects.all()
    serializer_class = LeaseSerializer
