from django.http import Http404
from django.conf import settings
from rest_framework.views import APIView
from rest_framework.response import Response
from rest_framework import status

from interface_openstack.implementation.vm.instance import (
                                                        OSVirtualMachineManager
                                                        )
from instance.models import Instance, Flavor, Lease
from instance.serializers import InstanceSerializer, FlavorSerializer


class InstanceList(APIView):
    def get(self, request, format=None):
        instances = Instance.objects.all()
        return Response(InstanceSerializer(instances, many=True).data)

    def post(self, request, format=None):
        data = request.data
        template = ImageTemplate.objects.get(pk=data["template"])
        flavor = Flavor.objects.get(pk=data["flavor"])
        lease = Lease.objects.get(pk=data["lease"])

        newInstance = Instance.create_instance_from_template(
            params={"name": data["name"],
                    "description": data["description"],
                    "access_method": data["access"],
                    "system": data["system"],
                    },
            lease=lease,
            networks=template.networks,
            template=template,
            flavor=flavor,
            owner=request.user,
            disks=template.disks
        )
        return Response(newInstance.pk)


class InstanceDetail(APIView):
    """
    Retrieve, update or delete a snippet instance.
    """

    def get_object(self, pk):
        try:
            return Instance.objects.get(pk=pk)
        except Instance.DoesNotExist:
            raise Http404

    def get(self, request, pk, format=None):
        instance = self.get_object(pk)
        instanceDict = InstanceSerializer(instance).data

        interface = OSVirtualMachineManager(settings.CONNECTION)
        remoteInstance = interface.get_vm(instance.remote_id)
        remoteInstanceDict = remoteInstance.__dict__

        merged_dict = {**instanceDict, **remoteInstanceDict}

        return Response(merged_dict)

    def put(self, request, pk, format=None):
        instance = self.get_object(pk)
        serializer = InstanceSerializer(instance, data=request.data)
        if serializer.is_valid():
            serializer.save()
            return Response(serializer.data)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    def delete(self, request, pk, format=None):
        instance = self.get_object(pk)
        instance.delete()
        return Response(status=status.HTTP_204_NO_CONTENT)


class FlavorListView(APIView):
    """
    Create, update or delete a flavor.
    """

    def get(self, request, format=None):
        flavors = Flavor.objects.all()
        return Response(FlavorSerializer(flavors, many=True).data)

    def post(self, request, format=None):
        data = request.data
        new_flavor = Flavor.create(name=data["name"],
                                   description=data["description"],
                                   ram=data["ram"],
                                   vcpu=data["vcpu"],
                                   initial_disk=data["initial_disk"],
                                   priority=data["priority"])

        return Response(new_flavor.pk)
