from django.db import models
from django.conf import settings
from django.contrib.auth.models import User
from django.utils import timezone
from datetime import timedelta
from image.models import Disk
from interface_openstack.implementation.vm.instance import (
        OSVirtualMachineManager
      )


ACCESS_METHODS = tuple(
    [(key, val[0]) for key, val in settings.VM_ACCESS_PROTOCOLS.items()]
)


class Lease(models.Model):
    """ Users can use the virtual machine until the lease dates.
        After suspend interval the vm suspends and after delete it will be
        destroyed
    """
    name = models.CharField(blank=True, max_length=100)
    description = models.CharField(blank=True, max_length=100)
    suspend_interval_in_sec = models.IntegerField(blank=True, null=True)
    delete_interval_in_sec = models.IntegerField(blank=True, null=True)


class Flavor(models.Model):
    """ Flavors are reasource packages, contains all information about
        resources accociated with the virtual machine
    """
    name = models.CharField(blank=True, max_length=100)
    description = models.CharField(blank=True, max_length=200)
    remote_id = models.CharField(
            max_length=100, help_text="ID of the instance on the backend"
        )
    ram = models.IntegerField(blank=True, null=True)
    vcpu = models.IntegerField(blank=True, null=True)
    initial_disk = models.IntegerField(blank=True, null=True)
    priority = models.IntegerField(blank=True, null=True)

    @classmethod
    def create(cls, name, description, ram=0, vcpu=0,
               initial_disk=0, priority=0):
        interface = OSVirtualMachineManager(settings.CONNECTION)
        remote_flavor = interface.create_flavor(name, ram, vcpu, initial_disk)
        flavor = cls(name=name, description=description,
                     remote_id=remote_flavor.id, ram=ram, vcpu=vcpu,
                     initial_disk=initial_disk, priority=priority)
        flavor.save()

        return flavor


class Instance(models.Model):
    """Virtual machine instance.
    """

    name = models.CharField(max_length=100,
                            help_text="Human readable name of instance")

    remote_id = models.CharField(
        max_length=100, help_text="ID of the instance on the backend"
    )
    description = models.TextField(
        blank=True, help_text="The description of the instance"
    )
    access_method = models.CharField(
        max_length=10, choices=ACCESS_METHODS,
        help_text="Primary remote access method"
    )
    system = models.CharField(max_length=50, help_text="Operating system type")
    password = models.CharField(
        max_length=50, help_text="Original password of the instance"
    )
    time_of_suspend = models.DateTimeField(
        blank=True,
        help_text="After this point in time, the instance will be suspended"
    )
    time_of_delete = models.DateTimeField(
        blank=True,
        help_text="After this point in time, the instance will be deleted!"
    )
    deleted = models.BooleanField(
        help_text="Indicates if the instance is ready for garbage collection",
        default=False,
    )

    # image = models.ForeignKey(TemplateImage, related_name="vm", null=True,
    #                           help_text="The base image of the vm")
    #
    # snapshot = models.ForeignKey(Snapshot, related_name="vm", null=True,
    #                              help_text="The base snapshot of the vm")

    disks = models.ManyToManyField(Disk, related_name="instance",
                                   help_text="Disks attached to instance",
                                   verbose_name="disks")

    owner = models.ForeignKey(User, on_delete="CASCADE", null=True)
    flavor = models.ForeignKey(Flavor, help_text="Reasources given to the vm",
                               verbose_name="flavor", on_delete="CASCADE",
                               related_name='instances')
    lease = models.ForeignKey(Lease, on_delete="CASCADE",
                              related_name='instances')

    @classmethod
    def create(cls, lease, owner, flavor, remote_id, params):
        inst = cls(lease=lease, flavor=flavor, owner=owner,
                   remote_id=remote_id, **params)
        inst.full_clean()
        inst.save()

        return inst

    @classmethod
    def create_instance_from_template(cls, params, template, owner, lease,
                                      disks, networks, flavor):
        # TODO: attach disks when the remote instance created
        interface = OSVirtualMachineManager(settings.CONNECTION)
        remote_inst = interface.create_vm_from_template(params["name"],
                                                        template.remote_ID,
                                                        flavor.remote_id,
                                                        networks,
                                                        )
        remote_id = remote_inst.id
        new_inst = cls.create(lease, owner, flavor, remote_id, params)

        return new_inst

    def clean(self, *args, **kwargs):
        self.time_of_suspend, self.time_of_delete = self.get_renew_times()
        super(Instance, self).clean(*args, **kwargs)

    def get_renew_times(self, lease=None):
        """Returns new suspend and delete times if renew would be called.
        """
        if lease is None:
            lease = self.lease
        return (
            timezone.now() + timedelta(
                                seconds=lease.suspend_interval_in_sec),
            timezone.now() + timedelta(
                                seconds=lease.delete_interval_in_sec)
            )
