from django.db import models
from django.contrib.auth.models import User
from django.conf import settings

from interface_openstack.implementation.vm.instance import OSVirtualMachineManager
from interface_openstack.implementation.image.openstack_image_manager import OpenstackImageManager


class Disk(models.Model):

    """A virtual disk.
    """

    name = models.CharField(
        blank=True, max_length=100, verbose_name="name", help_text="Name of the disk"
    )
    remote_id = models.CharField(
        max_length=40,
        unique=True,
        verbose_name="remote_ID",
        help_text="ID, which helps access the disk",
    )


class Image(models.Model):

    """A virtual image.
    """

    name = models.CharField(
        max_length=100,
        verbose_name="name",
        help_text="Human readable name of image."
    )
    description = models.TextField(
        verbose_name="description",
        blank=True,
        help_text="Description of the image."
    )
    remote_id = models.CharField(
        max_length=40,
        unique=True,
        verbose_name="remote_ID",
        help_text="ID, which helps access the image."
    )
    created_at = models.DateTimeField(
        auto_now_add=True,
        editable=False,
        help_text="Date, when the image created."
    )
    uploaded_by_user = models.BooleanField(
        default=True,
        editable=False,
        help_text="The field is false if the image created from instance"
    )
    created_by = models.ForeignKey(
        User,
        on_delete=models.DO_NOTHING,
        related_name="created_images",
        help_text="The user, who create the image"
    )

    @classmethod
    def create(cls, name, description, remote_id, uploaded_by_user, created_by):
        inst = cls(name=name, description=description, uploaded_by_user=uploaded_by_user,
                   remote_id=remote_id, created_by=created_by)
        inst.full_clean()
        inst.save()

        return inst


    @classmethod
    def create_from_instance(cls, user, instance, description):
        interface = OSVirtualMachineManager(settings.CONNECTION)
        remote_image = interface.create_image(instance.remote_id)
        remote_id = remote_image.id
        name = remote_image.name
        new_image = cls.create(
            name=name,
            remote_id=remote_id,
            created_by=user,
            uploaded_by_user=False,
            description=description
        )
        return new_image
    
    @classmethod
    def create_from_user(cls, description, file_format, image_file, name, user):
        interface = OpenstackImageManager(settings.CONNECTION)
        remote_image = interface.upload_file(name=name, path=image_file.temporary_file_path(), format=file_format)
        new_image = cls.create(
            name=name,
            remote_id=remote_image.id,
            created_by=user,
            uploaded_by_user=True,
            description=description
        )
        return new_image
