from instance.serializers import InstanceSerializer, FlavorSerializer, LeaseSerializer
from django.http import Http404
from rest_framework.viewsets import ViewSet, ModelViewSet
from rest_framework.response import Response
from rest_framework import status
from rest_framework.decorators import action

from template.serializers import InstanceFromTemplateSerializer
from instance.models import Instance, Flavor, Lease
from template.models import ImageTemplate
from template.serializers import ImageTemplateModelSerializer
from authorization.mixins import AuthorizationMixin


authorization = {
    "list": {"filter": ["use_instance"]},
    "create": {"model": ["instance.create_instance"]},
    "retrieve": {"object": ["use_instance"]},
    "update": {"object": ["use_instance"]},
    "destroy": {"object": ["administer_instance"]},
    "template": {"model": ["create_template_from_instance"],
                 "object": ["use_instance"]},
    "start": {"object": ["operate_instance"]},
    "stop": {"object": ["operate_instance"]},
    "suspend": {"object": ["operate_instance"]},
    "wake_up": {"object": ["operate_instance"]},
    "reset": {"object": ["operate_instance"]},
    "reboot": {"object": ["operate_instance"]},
}

update_actions = [
    "change_name",
    "change_description",
    "renew",
    "change_lease",
    "change_flavor",
    "attach_disk",
    "resize_disk",
    "add_permission",
    "remove_permission",
    "open_port",
    "close_port",
    "add_network",
    "remove_network",
    "new_password",
]


class InstanceViewSet(AuthorizationMixin, ViewSet):

    authorization = authorization

    def get_object(self, pk):
        try:
            return Instance.objects.get(pk=pk)
        except Instance.DoesNotExist:
            raise Http404

    def get_merged_object(self, pk):
        instance = self.get_object(pk)
        instanceDict = InstanceSerializer(instance).data
        remoteInstance = instance.get_remote_instance()
        remoteInstanceDict = {
            "remote_id": remoteInstance.id,
            "status": remoteInstance.status,
            "disks": remoteInstance.disks,
            "addresses": remoteInstance.addresses,
        }
        return({**instanceDict, **remoteInstanceDict})

    def list(self, request):
        instances = self.get_objects_with_perms(request.user, "list", Instance)
        return Response(InstanceSerializer(instances, many=True).data)

    def create(self, request):
        if not self.has_perms_for_model(request.user, 'create'):
            return Response({"error": "No permission to create Virtual Machine."},
                            status=status.HTTP_401_UNAUTHORIZED)
        params = request.data
        template = ImageTemplate.objects.get(pk=params["template"])
        flavor = Flavor.objects.get(pk=params["flavor"]) if "flavor" in params else template.flavor
        lease = Lease.objects.get(pk=params["lease"]) if "lease" in params else template.lease
        newInstance = Instance.create_instance_from_template(
            params={"name": params["name"],
                    "description": params["description"],
                    "access_method": params["access"],
                    "system": params["system"],
                    },
            lease=lease,
            networks=[{"uuid": template.network_id}],
            template=template,
            flavor=flavor,
            owner=request.user,
            disks=None
        )
        return Response(InstanceSerializer(newInstance).data)

    def retrieve(self, request, pk):
        instance = self.get_object(pk)
        if not self.has_perms_for_object(request.user, 'retrieve', instance):
            return Response({"error": "No permission to access the Virtual Machine."},
                            status=status.HTTP_401_UNAUTHORIZED)
        mergedInstance = self.get_merged_object(pk)
        return Response(mergedInstance)

    def update(self, request, pk, format=None):
        if request.data["action"] in update_actions:
            instance = self.get_object(pk)
            if not self.has_perms_for_object(request.user, 'update', instance):
                return Response({"error": "No permission to access the Virtual Machine."},
                                status=status.HTTP_401_UNAUTHORIZED)
            action = request.data["action"]
            if action == "change_name":
                instance.change_name(request.data["name"])
            elif action == "change_description":
                instance.change_description(request.data["description"])
            elif action == "renew":
                instance.renew()
            elif action == "change_lease":
                lease = Lease.objects.get(pk=request.data["lease"])
                instance.renew(lease)
            elif action == "change_flavor":
                pass
            elif action == "attach_disk":
                pass
            elif action == "resize_disk":
                pass
            elif action == "add_permission":
                pass
            elif action == "remove_permission":
                pass
            elif action == "open_port":
                pass
            elif action == "close_port":
                pass
            elif action == "add_network":
                pass
            elif action == "remove_network":
                pass
            elif action == "new_password":
                pass

            instanceDict = InstanceSerializer(instance).data
            remoteInstance = instance.get_remote_instance()
            remoteInstanceDict = remoteInstance.__dict__
            merged_dict = {"db": instanceDict, "openstack": remoteInstanceDict}
            return Response(merged_dict)
        else:
            return Response({"error": "Unknown update action."}, status=status.HTTP_400_BAD_REQUEST)

    def destroy(self, request, pk, format=None):
        instance = self.get_object(pk)
        if not self.has_perms_for_object(request.user, 'destroy', instance):
            return Response({"error": "No permission to destroy the Virtual Machine."},
                            status=status.HTTP_401_UNAUTHORIZED)
        instance.delete()
        return Response(status=status.HTTP_204_NO_CONTENT)

    @action(detail=True, methods=["post"])
    def template(self, request, pk):
        instance = self.get_object(pk)
        if not self.has_perms_for_model(request.user, 'template'):
            return Response({"error": "No permission to create template from instance."},
                            status=status.HTTP_401_UNAUTHORIZED)
        if not self.has_perms_for_object(request.user, 'template', instance):
            return Response({"error": "No permission to access the Virtual Machine."},
                            status=status.HTTP_401_UNAUTHORIZED)
        serializer = InstanceFromTemplateSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        data = serializer.validated_data
        new_template = ImageTemplate.create_from_instance(data["name"],
                                                          data["description"],
                                                          instance,
                                                          request.user)
        serializer = ImageTemplateModelSerializer(instance=new_template)
        return Response(serializer.data)

    @action(detail=True, methods=["POST"])
    def actions(self, request, pk):
        instance = self.get_object(pk)
        if not self.has_perms_for_object(request.user, request.data["action"], instance):
            return Response({"error": "No permission to use this action on the VM."},
                            status=status.HTTP_401_UNAUTHORIZED)
        success = instance.execute_common_action(action=request.data["action"])
        return Response(success)


class FlavorViewSet(ViewSet):
    """
    Create, update or delete a flavor.
    """

    def get_object(self, pk):
        try:
            return Flavor.objects.get(pk=pk)
        except Flavor.DoesNotExist:
            raise Http404

    def list(self, request, format=None):
        flavors = Flavor.objects.all()
        return Response(FlavorSerializer(flavors, many=True).data)

    def create(self, request, format=None):
        data = request.data
        new_flavor = Flavor.create(name=data["name"],
                                   description=data["description"],
                                   ram=data["ram"],
                                   vcpu=data["vcpu"],
                                   initial_disk=data["initial_disk"],
                                   priority=data["priority"])
        return Response(new_flavor.pk)

    def update(self, request, pk):
        return Response(status=status.HTTP_400_BAD_REQUEST)

    def partial_update(self, request, pk):
        return Response(status=status.HTTP_400_BAD_REQUEST)

    def destroy(self, request, pk):
        flavor = self.get_object(pk)
        flavor.delete()
        return Response(status=status.HTTP_204_NO_CONTENT)


class LeaseViewSet(ModelViewSet):
    queryset = Lease.objects.all()
    serializer_class = LeaseSerializer
