import libvirt
import sys
import uuid
from jinja2 import Environment, select_autoescape, FileSystemLoader
import subprocess

from utils import fwd_port, show_ip

try:
    conn = libvirt.open("qemu:///system")
except libvirt.libvirtError:
    print('Failed to open connection to the hypervisor')
    sys.exit(1)

file_loader = FileSystemLoader('templates')

env = Environment(autoescape=select_autoescape(
    enabled_extensions=('xml'),
    default_for_string=True,
), loader=file_loader)

params = {
    "name": "demo",
    "uuid": uuid.uuid4(),
    "ram": 4, # in GB
    "vcpu": 2,
    "disk": "/data/vms/s1/s1-ubuntu.qcow2",
    "boot_iso": True,
    "iso": "/data/img/ubuntu-22.04.2-live-server-amd64.iso",
    "vgpu": True,
    "vgpu_type": "nvidia-1",
    "mdev_bus": "0000\:01\:00.0"
}

template = env.get_template("demovm.xml")

vmxml = template.render(
    **params
)

print(vmxml)

input("Correct? ")

id = params["uuid"]
bus = params["mdev_bus"]
vgpu = params["vgpu_type"]

if params["vgpu"]:
    print("Allocating VGPU instance for the VM")
    try:
        subprocess.run([f"sudo bash create_vgpu.sh {id} {bus} {vgpu}"], check=True, shell=True, stderr=subprocess.STDOUT)
    except subprocess.CalledProcessError as e:
        raise RuntimeError("command '{}' return with error (code {}): {}".format(e.cmd, e.returncode, e.output))

try:
    dom = conn.defineXML(vmxml)
    if not dom:
        raise SystemExit("Failed to define a domain from an XML definition")
    print("Start VM")
    if dom.create() < 0:
        raise SystemExit("Can not boot guest domain")
except libvirt.libvirtError:
    print('Failed to find the main domain')
    sys.exit(1)

print("Domain 0: id %d running %s" % (dom.ID(), dom.OSType()))
print(dom.info())

print("Type stop, to shutdown vm!")
cmd = input()
while cmd != "stop":
    cmdarr = cmd.split()
    if cmdarr[0] == "fwd":
        fwd_port(dom, int(cmdarr[1]), int(cmdarr[2]))
    elif cmdarr[0] == "showip":
        show_ip(dom)
    cmd = input()

try:
    print("Stop VM")
    if dom.destroy() < 0:
        raise SystemExit("Can not stop guest domain")
except libvirt.libvirtError:
    print('Failed to find the main domain')
    sys.exit(1)    

if params["vgpu"]:
    print("Deallocating VGPU instance from the VM")
    try:
        subprocess.run([f"sudo bash remove_vgpu.sh {id} {bus} {vgpu}"], check=True, shell=True, stderr=subprocess.STDOUT)
    except subprocess.CalledProcessError as e:
        raise RuntimeError("command '{}' return with error (code {}): {}".format(e.cmd, e.returncode, e.output))


conn.close()