# circle-agent-install.ps1 (v1.1)
# Installs and starts CIRCLE agent + watchdog, and registers circle-notify.exe as logon task
# Run from elevated PowerShell as cloud user
# ASCII-only comments

param(
  [switch]$WhatIf
)

$ErrorActionPreference = "Stop"

function Say($msg="") { Write-Host $msg }

function Fail($msg) {
  Write-Host "ERROR: $msg" -ForegroundColor Red
  exit 1
}

function Require-Admin {
  $id = [Security.Principal.WindowsIdentity]::GetCurrent()
  $p  = New-Object Security.Principal.WindowsPrincipal($id)
  if (-not $p.IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)) {
    Fail "Run this script as Administrator."
  }
}

function Run-Step($desc, [ScriptBlock]$action) {
  if ($WhatIf) {
    Say ("[WHATIF] {0}" -f $desc)
    return
  }

  Say ("[DO] {0}" -f $desc)
  try {
    & $action
  } catch {
    Fail ("{0} failed: {1}" -f $desc, $_.Exception.Message)
  }
}

function Check-File($path) {
  if (-not (Test-Path $path)) { Fail "Missing file: $path" }
}

function Check-Dir($path) {
  if (-not (Test-Path $path)) { Fail "Directory not found: $path" }
}

function Check-ServiceRunning($name) {
  $s = Get-Service -Name $name -ErrorAction SilentlyContinue
  if (-not $s) { Fail "Service not found: $name" }
  if ($s.Status -ne "Running") {
    Fail "Service $name is not running (state=$($s.Status))"
  }
}

Require-Admin

$baseDir     = "C:\circle"
$agentExe    = Join-Path $baseDir "circle-agent.exe"
$watchdogExe = Join-Path $baseDir "circle-watchdog.exe"
$notifyExe   = Join-Path $baseDir "circle-notify.exe"
$taskName    = "CIRCLE circle-notify (Logon)"

Say "=== CIRCLE Agent install ==="
Say ("User: {0}" -f $env:USERNAME)
Say ("Working dir: {0}" -f $baseDir)
Say ("Mode: {0}" -f $(if ($WhatIf) { "WhatIf (dry-run)" } else { "Apply" }))
Say ""

# --- Preconditions ---
Check-Dir $baseDir
Check-File $agentExe
Check-File $watchdogExe
Check-File $notifyExe

# Show what would run (useful in WhatIf)
Say "Plan:"
Say ("  - cd {0}" -f $baseDir)
Say ("  - {0} --startup auto install" -f $agentExe)
Say ("  - {0} --startup auto install" -f $watchdogExe)
Say ("  - {0} start" -f $agentExe)
Say ("  - {0} start" -f $watchdogExe)
Say ("  - Register Scheduled Task: {0} -> {1}" -f $taskName, $notifyExe)
Say ""

Run-Step "Set location to $baseDir" { Set-Location $baseDir }

# --- Install services ---
Run-Step "Install circle-agent service (startup auto)" {
  & $agentExe --startup auto install | Out-Null
}

Run-Step "Install circle-watchdog service (startup auto)" {
  & $watchdogExe --startup auto install | Out-Null
}

# --- Start services ---
Run-Step "Start circle-agent service" {
  & $agentExe start | Out-Null
}

Run-Step "Start circle-watchdog service" {
  & $watchdogExe start | Out-Null
}

# --- Verify services (skip on WhatIf) ---
if (-not $WhatIf) {
  Check-ServiceRunning "circle-agent"
  Check-ServiceRunning "circle-watchdog"
} else {
  Say "[WHATIF] Skip service running checks (no changes applied)."
}

# --- Register circle-notify as logon task ---
Run-Step "Register Scheduled Task '$taskName' for circle-notify.exe" {
  $action  = New-ScheduledTaskAction -Execute $notifyExe
  $trigger = New-ScheduledTaskTrigger -AtLogOn

  # Run only when user is logged on (no password prompt)
  $principal = New-ScheduledTaskPrincipal -UserId $env:USERNAME -LogonType Interactive -RunLevel Highest

  $settings = New-ScheduledTaskSettingsSet `
    -MultipleInstances IgnoreNew `
    -ExecutionTimeLimit (New-TimeSpan -Seconds 0) `
    -RestartCount 5 `
    -RestartInterval (New-TimeSpan -Minutes 3)

  $task = New-ScheduledTask -Action $action -Trigger $trigger -Principal $principal -Settings $settings
  Register-ScheduledTask -TaskName $taskName -InputObject $task -Force | Out-Null
}

# --- Verify task (skip on WhatIf) ---
if (-not $WhatIf) {
  $task = Get-ScheduledTask -TaskName $taskName -ErrorAction SilentlyContinue
  if (-not $task) { Fail "Scheduled Task '$taskName' not found after registration" }
} else {
  Say "[WHATIF] Skip task verification (no changes applied)."
}

Say ""
if ($WhatIf) {
  Say "DRY-RUN OK: Preconditions passed; no changes were made."
} else {
  Say "SUCCESS:"
  Say " - circle-agent service installed and running"
  Say " - circle-watchdog service installed and running"
  Say " - circle-notify scheduled task registered (logon, restart-on-failure)"
  Say ""
  Say "NOTE: User logoff/logon required for circle-notify.exe to start."
}
